/* ports.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Generic I/O ports provided by the AVR microcontroller.

// Imports ===================================================================



use core::any::Any;
use ufmt::derive::uDebug;
use oxide_macros::Persist;
use avr_oxide::hal::generic::callback::IsrCallback;

// Declarations ==============================================================
/**
 * Input/output mode for a pin
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum PinMode {
  /// Pin is configured as output
  Output,
  /// Pin is configured as an input with internal pullup enabled
  InputPullup,
  /// Pin is configured as an input with internal pullup disabled
  InputFloating
}

/**
 * Interrupt generation mode for a pin
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum InterruptMode {
  /// Do not generate interrupts
  Disabled,
  /// Interrupt on both rising and falling edges
  BothEdges,
  /// Interrupt on rising edge
  RisingEdge,
  /// Interrupt on falling edge
  FallingEdge,
  /// Interrupt while input is low
  LowLevel
}

/**
 * Identifies a particular pin as the source of an event at runtime.
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum PinIdentity {
  /// Port A, pin number (u8)
  PortA(u8),
  /// Port A, pin number (u8)
  PortB(u8),
  /// Port A, pin number (u8)
  PortC(u8),
  /// Port A, pin number (u8)
  PortD(u8),
  /// Port A, pin number (u8)
  PortE(u8),
  /// Port A, pin number (u8)
  PortF(u8)
}

/**
 * Callback called by a pin when it generates an interrupt.  The callback
 * is given the new state read from the pin (true = high).
 * The callback should return a boolean:
 * * true:  Continue generating interrupts for this pin
 * * false: Disable generating interrupts for this pin
 *
 * Note: The callback runs *within the interrupt context* - be careful about
 * using mutual exclusion where necessary, and *DO NOT DO HEAVY PROCESSING
 * IN THE CALLBACK*.
 */
pub type PinIsrFunction<P> = fn(&'static P,PinIdentity,bool,Option<*const dyn Any>) -> ();

pub type PinIsrCallback<P> = IsrCallback<PinIsrFunction<P>,()>;

pub trait Pin {
  /// Set the input/output mode for this pin
  fn set_mode(&self, mode: PinMode);

  /// Toggle this pin's output level
  fn toggle(&self);

  /// Set the pin's output to high
  fn set_high(&self);

  /// Set the pin's output to low
  fn set_low(&self);

  /// Set the pin's output according to the boolean (true == high, false == low)
  fn set(&self, high: bool);

  /// Get the pin's input; true == high, false == low.
  fn get(&self) -> bool;

  /// Set when this pin will generate interrupts
  fn set_interrupt_mode(&self, mode: InterruptMode);

  /// Listen for interrupts on this pin and call the given handler callback
  /// when one occurs.
  fn listen(&'static self, handler: PinIsrCallback<Self>);
}

pub trait Port<P>
where
  P: Pin
{
  fn pin_instance(&self, pin: u8) -> &P;
}

// Code ======================================================================
#[cfg(target_arch="avr")]
pub mod base {
  use avr_oxide::hal::generic::port::InterruptMode;
  use avr_oxide::{v_read, v_write};

  pub trait AtmelPortControl {
    /// Set this pin as an output.
    fn enable_output(&mut self, p: u8);
    /// Disable output on this pin
    fn disable_output(&mut self, p: u8);
    /// Enable internal pullup on this pin
    fn enable_pullup(&mut self, p: u8);
    /// Disable internal pullup on this pin
    fn disable_pullup(&mut self, p: u8);
    /// Set a pin's output state high
    fn set_high(&mut self, p: u8);
    /// Set a pin's output state low
    fn set_low(&mut self, p: u8);
    /// Toggle a pin's input state
    fn toggle(&mut self, p: u8);
    /// Get a pin's state
    fn get(&self, p: u8) -> bool;
    /// Set the mode for interrupt generation for a pin
    fn set_interrupt_mode(&mut self, p: u8, mode: InterruptMode);
    /// True iff the given pin has generated an interrupt
    fn interrupted(&self, p: u8) -> bool;
    /// Clear the interrupt flag for all pins on this port
    fn clear_interrupts(&mut self);
    /// Return a tuple (pin,state) of pins generating an interrupt.  Called
    /// from the ISR, so make it quick.
    fn isr_get_interrupting_pins(&mut self)->(u8,u8);
  }

  impl InterruptMode {
    #[inline(always)]
    fn to_pinctrl_isc(&self) -> u8 {
      match self {
        InterruptMode::Disabled => 0x00,
        InterruptMode::BothEdges => 0x01,
        InterruptMode::RisingEdge => 0x02,
        InterruptMode::FallingEdge => 0x03,
        InterruptMode::LowLevel => 0x05
      }
    }

    #[inline(always)]
    fn pinctrl_mask() -> u8 {
      0b00000111
    }
  }

  /**
   * The AVR port control register block.  The structure provided by the
   * auto-generated avr-device crate is just horrible, so we use our own
   * rendition.  This allows accessing pin control by index, and also avoids
   * having different types for every port.
   */
  #[repr(C)]
  pub struct AvrPortRegisterBlock {
    pub(crate) dir: u8,
    pub(crate) dir_set: u8,
    pub(crate) dir_clr: u8,
    pub(crate) dir_tgl: u8,
    pub(crate) out: u8,
    pub(crate) out_set: u8,
    pub(crate) out_clr: u8,
    pub(crate) out_tgl: u8,
    pub(crate) inp: u8,
    pub(crate) intflags: u8,
    pub(crate) portctrl: u8,
    pub(crate) reserved: [u8; 5],
    pub(crate) pinctrl: [u8; 8]
  }

  /**
   * The older ATmegas like the '328P have a much simpler port control
   * block (if you can even call it that...)
   */
  #[repr(C)]
  pub struct AvrSimplePortRegisterBlock<const PCIEN: u16, const PCICR: u16, const PCIFR: u16, const PCICRMASK: u8> {
    pub(crate) pin: u8,
    pub(crate) ddr: u8,
    pub(crate) port: u8
  }


  impl AtmelPortControl for AvrPortRegisterBlock {
    #[inline(always)]
    fn enable_output(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.dir_set as *mut u8, 0x01 << p);
      }
    }
    #[inline(always)]
    fn disable_output(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.dir_clr as *mut u8, 0x01 << p);
      }
    }

    fn enable_pullup(&mut self, p: u8) {
      unsafe {
        let pinctrl_reg = &mut self.pinctrl[p as usize] as *mut u8;

        let pinctrl = core::ptr::read_volatile(pinctrl_reg);

        core::ptr::write_volatile(pinctrl_reg, pinctrl | 0b00001000);
      }
    }

    fn disable_pullup(&mut self, p: u8) {
      unsafe {
        let pinctrl_reg = &mut self.pinctrl[p as usize] as *mut u8;

        let pinctrl = core::ptr::read_volatile(pinctrl_reg);

        core::ptr::write_volatile(pinctrl_reg, pinctrl & 0b11110111);
      }
    }

    #[inline(always)]
    fn set_high(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.out_set as *mut u8, 0x01 << p);
      }
    }
    #[inline(always)]
    fn set_low(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.out_clr as *mut u8, 0x01 << p);
      }
    }
    #[inline(always)]
    fn toggle(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.out_tgl as *mut u8, 0x01 << p);
      }
    }

    #[inline(always)]
    fn get(&self, p: u8) -> bool {
      unsafe {
        core::ptr::read_volatile(&self.inp as *const u8) & (0x01 << p) != 0
      }
    }

    fn set_interrupt_mode(&mut self, p: u8, mode: InterruptMode) {
      unsafe {
        let pinctrl_reg = &mut self.pinctrl[p as usize] as *mut u8;

        let pinctrl = core::ptr::read_volatile(pinctrl_reg);

        core::ptr::write_volatile(pinctrl_reg,
                                  (pinctrl & !InterruptMode::pinctrl_mask()) | mode.to_pinctrl_isc());
      }
    }

    #[inline(always)]
    fn interrupted(&self, p: u8) -> bool {
      unsafe {
        core::ptr::read_volatile(&self.intflags as *const u8) & (0x01 << p) != 0
      }
    }

    #[inline(always)]
    fn clear_interrupts(&mut self) {
      unsafe {
        core::ptr::write_volatile(&mut self.intflags as *mut u8, 0xff);
      }
    }

    #[inline(always)]
    fn isr_get_interrupting_pins(&mut self) -> (u8,u8) {
      unsafe {
        let intflags:u8 = v_read!(u8, self.intflags);
        let state:u8    = v_read!(u8, self.inp);

        (intflags,state)
      }
    }
  }

  impl<const PCIEN: u16, const PCICR: u16, const PCIFR:u16,  const PCICRMASK: u8> AtmelPortControl for AvrSimplePortRegisterBlock<PCIEN,PCICR,PCIFR,PCICRMASK> {
    fn enable_output(&mut self, p: u8) {
      unsafe {
        v_write!(u8, self.ddr, v_read!(u8, self.ddr) | 0x01 << p);
      }
    }

    fn disable_output(&mut self, p: u8) {
      unsafe {
        v_write!(u8, self.ddr, v_read!(u8, self.ddr) ^ 0x01 << p);
      }
    }

    fn enable_pullup(&mut self, p: u8) {
      unsafe {
        v_write!(u8, self.port, v_read!(u8, self.port) | 0x01 << p);
      }
    }

    fn disable_pullup(&mut self, p: u8) {
      unsafe {
        v_write!(u8, self.port, v_read!(u8, self.port) | 0x01 << p);
      }
    }

    fn set_high(&mut self, p: u8) {
      unsafe {
        v_write!(u8, self.port, v_read!(u8, self.port) | 0x01 << p);
      }
    }

    fn set_low(&mut self, p: u8) {
      unsafe {
        v_write!(u8, self.port, v_read!(u8, self.port) | 0x01 << p);
      }
    }

    fn toggle(&mut self, p: u8) {
      unsafe {
        v_write!(u8, self.pin, 0x01 << p); // Writing 1 to the input toggles the output on '328p et al
      }
    }

    fn get(&self, p: u8) -> bool {
      unsafe {
        v_read!(u8, self.pin) & (0x01 << p) > 0
      }
    }

    fn set_interrupt_mode(&mut self, p: u8, mode: InterruptMode) {
      unsafe {
        let pcien = &mut *(PCIEN as *mut u8);
        let pcicr = &mut *(PCICR as *mut u8);

        match mode {
          // @todo actually change the global level/edge mode
          InterruptMode::Disabled => {
            v_write!(u8, *pcien, v_read!(u8, *pcien) ^ 0x01 << p);
          }
          InterruptMode::BothEdges => {
            v_write!(u8, *pcicr, v_read!(u8, *pcicr) | PCICRMASK);
            v_write!(u8, *pcien, v_read!(u8, *pcien) | 0x01 << p);
          },
          InterruptMode::RisingEdge => {
            v_write!(u8, *pcicr, v_read!(u8, *pcicr) | PCICRMASK);
            v_write!(u8, *pcien, v_read!(u8, *pcien) | 0x01 << p);
          },
          InterruptMode::FallingEdge => {
            v_write!(u8, *pcicr, v_read!(u8, *pcicr) | PCICRMASK);
            v_write!(u8, *pcien, v_read!(u8, *pcien) | 0x01 << p);
          },
          InterruptMode::LowLevel => {
            v_write!(u8, *pcicr, v_read!(u8, *pcicr) | PCICRMASK);
            v_write!(u8, *pcien, v_read!(u8, *pcien) | 0x01 << p);
          }
        }
      }
    }

    /**
     * Indicate if a pin generated an interrupt.  We can't actually tell, so
     * we're just gonna say yes for all of them.
     */
    fn interrupted(&self, _p: u8) -> bool {
      true
    }

    /**
     * Clear interrupts.  We don't actually need to do this on the '328 because
     * they are cleared automatically when the ISR completes, but no harm done
     * by forcing the issue.
     */
    fn clear_interrupts(&mut self) {
      unsafe {
        let pcifr = &mut *(PCIFR as *mut u8);

        v_write!(u8, *pcifr, PCICRMASK)
      }
    }

    /**
     * Indicate which pin generated an interrupt.  Unfortunately the '328
     * doesn't make this easy for us, so we just say "all of 'em".
     */
    fn isr_get_interrupting_pins(&mut self) -> ( u8, u8 ) {
      unsafe {
        ( 0xff, v_read!(u8, self.pin) )
      }
    }
  }

  #[doc(hidden)]
  #[macro_export]
  macro_rules! atmel_port_tpl {
    ($ref:expr, $portsrc:expr, $implementation:ty, $isr:ident) => {
      use core::any::Any;

      use avr_oxide::hal::generic::port::{PinMode,Pin,PinIdentity,InterruptMode,PinIsrCallback};
      use avr_oxide::hal::generic::port::base::{AtmelPortControl};
      use avr_oxide::isr_cb_invoke;

      pub type PortImpl = $implementation;
      pub type PinImpl = AtmelPin;

      /**
       * A single pin instance.  This is just a wrapper for a reference to
       * the pin number and a place to stash the callback function that will
       * be called when we have an interrupt.
       */
      pub struct AtmelPin {
        n: u8,
        handler: PinIsrCallback<Self>
      }

      /**
       * Implementation of the Pin API for our Atmel PINs.
       */
      impl Pin for AtmelPin {
        #[inline(always)]
        fn set_mode(&self, mode: PinMode) {
          match mode {
            PinMode::Output => {
              instance().enable_output(self.n);
            },
            PinMode::InputPullup => {
              instance().enable_pullup(self.n);
              instance().disable_output(self.n);
            },
            PinMode::InputFloating => {
              instance().disable_pullup(self.n);
              instance().disable_output(self.n);
            }
          }
        }
        #[inline(always)]
        fn toggle(&self) {
          instance().toggle(self.n);
        }
        #[inline(always)]
        fn set_high(&self) {
          instance().set_high(self.n);
        }
        #[inline(always)]
        fn set_low(&self) {
          instance().set_low(self.n);
        }
        #[inline(always)]
        fn set(&self, high: bool) {
          match high {
            true  => instance().set_high(self.n),
            false => instance().set_low(self.n)
          }
        }
        #[inline(always)]
        fn get(&self) -> bool {
          instance().get(self.n)
        }
        fn set_interrupt_mode(&self, mode: InterruptMode) {
          instance().set_interrupt_mode(self.n, mode)
        }
        fn listen(&self, handler: PinIsrCallback<Self>){
          avr_oxide::hal::concurrency::interrupt::isolated(||{
            unsafe {
              PINS[self.n as usize].handler = handler;
            }
          });
        }
      }

      static mut INITIALISED: bool = false;

      /**
       * Get an instance of this port's register block.  Also does any
       * static initialisation required of the device the first time it is
       * called.
       */
      #[inline(always)]
      pub fn instance() -> &'static mut $implementation  {
        unsafe {
          // Not sure if this is a compiler bug or what, but our static
          // muts don't seem to be correctly initialised, so we need to
          // do this here instead
          //
          // ( @todo investigate - Probably I need to amend my boot code to call
          //         something to do the initialisation for me )
          if(!INITIALISED){
            for pin in 0..=7 {
              PINS[pin].n = pin as u8;
              PINS[pin].handler = PinIsrCallback::Nop(());
            }
            INITIALISED = true;
          }

          core::mem::transmute($ref)
        }
      }

      static mut PINS : [AtmelPin; 8] = [
        AtmelPin { n: 0, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 1, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 2, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 3, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 4, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 5, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 6, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 7, handler: PinIsrCallback::Nop(()) },
      ];


      /**
       * Return a single pin instance wrapper.
       */
      pub fn pin_instance(pin: u8) -> &'static mut AtmelPin {
        unsafe {
          &mut PINS[pin as usize]
        }
      }

      /**
       * Interrupt service routine.  Checks which pins caused the interrupt
       * and then calls the relevant callback routine.
       */
      #[no_mangle]
      pub unsafe extern "avr-interrupt" fn $isr() {
        avr_oxide::hal::concurrency::interrupt::isr(||{
          let (intflags, state) = instance().isr_get_interrupting_pins();

          let mut mask    = 0x01;

          for pin in 0..=7 {
            if (intflags & mask) > 0 {
              isr_cb_invoke!(PINS[pin].handler, &mut PINS[pin], $portsrc(pin as u8), (state & mask) > 0);
            }
            mask <<= 1;
          }

          instance().clear_interrupts();
        })
      }
    }
  }
}

#[cfg(not(target_arch="avr"))]
pub mod base {
  use avr_oxide::hal::generic::port::InterruptMode;

  pub trait AtmelPortControl {
    /// Set this pin as an output.
    fn enable_output(&mut self, p: u8);
    /// Disable output on this pin
    fn disable_output(&mut self, p: u8);
    /// Enable internal pullup on this pin
    fn enable_pullup(&mut self, p: u8);
    /// Disable internal pullup on this pin
    fn disable_pullup(&mut self, p: u8);
    /// Set a pin's output state high
    fn set_high(&mut self, p: u8);
    /// Set a pin's output state low
    fn set_low(&mut self, p: u8);
    /// Toggle a pin's input state
    fn toggle(&mut self, p: u8);
    /// Get a pin's state
    fn get(&self, p: u8) -> bool;
    /// Set the mode for interrupt generation for a pin
    fn set_interrupt_mode(&mut self, p: u8, mode: InterruptMode);
    /// True iff the given pin has generated an interrupt
    fn interrupted(&self, p: u8) -> bool;
    /// Clear the interrupt flag for all pins on this port
    fn clear_interrupts(&mut self);
  }

  /**
   * The AVR port control register block.  The structure provided by the
   * auto-generated avr-device crate is just horrible, so we use our own
   * rendition.  This allows accessing pin control by index, and also avoids
   * having different types for every port.
   */
  #[repr(C)]
  pub struct DummyPortRegisterBlock {
    pub(crate) pin: [bool; 8],
    pub(crate) o_en: [bool; 8],
    pub(crate) pu_en: [bool; 8]
  }



  impl AtmelPortControl for DummyPortRegisterBlock {
    fn enable_output(&mut self, p: u8) {
      println!("*** PORT: Enable output pin {}", p);
      self.o_en[p as usize] = true;
    }

    fn disable_output(&mut self, p: u8) {
      println!("*** PORT: Disable output pin {}", p);
      self.o_en[p as usize] = false;
    }

    fn enable_pullup(&mut self, p: u8) {
      println!("*** PORT: Enable pullup pin {}", p);
      self.pu_en[p as usize] = true;
    }

    fn disable_pullup(&mut self, p: u8) {
      println!("*** PORT: Disable pullup pin {}", p);
      self.pu_en[p as usize] = false;
    }

    fn set_high(&mut self, p: u8) {
      println!("*** PORT: Set HIGH pin {}", p);
      self.pin[p as usize] = true;
    }

    fn set_low(&mut self, p: u8) {
      println!("*** PORT: Set LOW pin {}", p);
      self.pin[p as usize] = false;
    }

    fn toggle(&mut self, p: u8) {
      let old = self.pin[p as usize];

      println!("*** PORT: TOGGLE pin {} ({} => {})", p, old, !old);

      self.pin[p as usize] = !old;
    }

    fn get(&self, p: u8) -> bool {
      let val = self.pin[p as usize];
      println!("*** PORT: GET pin {} ({})", p, val);
      val
    }

    fn set_interrupt_mode(&mut self, p: u8, _mode: InterruptMode) {
      println!("*** PORT: Set interrupt mode pin {}", p);
    }

    fn interrupted(&self, p: u8) -> bool {
      println!("*** PORT: Get interrupt flag pin {}", p);
      false
    }

    fn clear_interrupts(&mut self) {
      println!("*** PORT: Clear all interrupts");
    }


  }

  #[doc(hidden)]
  #[macro_export]
  macro_rules! atmel_port_tpl {
    ($ref:expr, $portsrc:expr, $implementation:ty, $isr:ident) => {
      use core::any::Any;

      use avr_oxide::hal::generic::port::{PinMode,Pin,PinIdentity,InterruptMode,PinIsrCallback};
      use avr_oxide::hal::generic::port::base::{AtmelPortControl,DummyPortRegisterBlock};

      pub type PortImpl = DummyPortRegisterBlock;
      pub type PinImpl = AtmelPin;

      /**
       * A single pin instance.  This is just a wrapper for a reference to
       * the pin number and a place to stash the callback function that will
       * be called when we have an interrupt.
       */
      pub struct AtmelPin {
        n: u8,
        handler: PinIsrCallback<Self>
      }

      /**
       * Implementation of the Pin API for our Atmel PINs.
       */
      impl Pin for AtmelPin {
        #[inline(always)]
        fn set_mode(&self, mode: PinMode) {
          match mode {
            PinMode::Output => {
              instance().enable_output(self.n);
            },
            PinMode::InputPullup => {
              instance().enable_pullup(self.n);
              instance().disable_output(self.n);
            },
            PinMode::InputFloating => {
              instance().disable_pullup(self.n);
              instance().disable_output(self.n);
            }
          }
        }
        #[inline(always)]
        fn toggle(&self) {
          instance().toggle(self.n);
        }
        #[inline(always)]
        fn set_high(&self) {
          instance().set_high(self.n);
        }
        #[inline(always)]
        fn set_low(&self) {
          instance().set_low(self.n);
        }
        #[inline(always)]
        fn set(&self, high: bool) {
          match high {
            true  => instance().set_high(self.n),
            false => instance().set_low(self.n)
          }
        }
        #[inline(always)]
        fn get(&self) -> bool {
          instance().get(self.n)
        }
        fn set_interrupt_mode(&self, mode: InterruptMode) {
          instance().set_interrupt_mode(self.n, mode)
        }
        fn listen(&self, handler: PinIsrCallback<Self>){
          avr_oxide::hal::concurrency::interrupt::isolated(||{
            unsafe {
              PINS[self.n as usize].handler = handler;
            }
          });
        }
      }

      static mut INITIALISED: bool = false;

      static mut INSTANCE : DummyPortRegisterBlock = DummyPortRegisterBlock {
        pin: [false; 8],
        o_en: [false; 8],
        pu_en: [false; 8]
      };

      /**
       * Get an instance of this port's register block.  Also does any
       * static initialisation required of the device the first time it is
       * called.
       */
      #[inline(always)]
      pub fn instance() -> &'static mut DummyPortRegisterBlock  {
        unsafe {
          // Not sure if this is a compiler bug or what, but our static
          // muts don't seem to be correctly initialised, so we need to
          // do this here instead
          //
          // ( @todo investigate - Probably I need to amend my boot code to call
          //         something to do the initialisation for me )
          if(!INITIALISED){
            for pin in 0..=7 {
              PINS[pin].n = pin as u8;
              PINS[pin].handler = PinIsrCallback::Nop(());
            }
            INITIALISED = true;
          }

          &mut INSTANCE
        }
      }

      static mut PINS : [AtmelPin; 8] = [
        AtmelPin { n: 0, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 1, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 2, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 3, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 4, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 5, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 6, handler: PinIsrCallback::Nop(()) },
        AtmelPin { n: 7, handler: PinIsrCallback::Nop(()) },
      ];


      /**
       * Return a single pin instance wrapper.
       */
      pub fn pin_instance(pin: u8) -> &'static mut AtmelPin {
        unsafe {
          &mut PINS[pin as usize]
        }
      }
    }
  }
}