/* serial.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! ATmega4809-specific serial device declaration/implementation

// Imports ===================================================================


// Declarations ==============================================================


#[cfg(feature="panicout_usart0")]
pub mod panicout {
  #[doc(hidden)]
  #[macro_export]
  macro_rules! panic_stdout { () => { avr_oxide::hal::atmega4809::serial::usart0::instance() }}
}

#[cfg(feature="panicout_usart1")]
pub mod panicout {
  #[doc(hidden)]
  #[macro_export]
  macro_rules! panic_stdout { () => { avr_oxide::hal::atmega4809::serial::usart1::instance() }}
}
#[cfg(feature="panicout_usart2")]
pub mod panicout {
  #[doc(hidden)]
  #[macro_export]
  macro_rules! panic_stdout { () => { avr_oxide::hal::atmega4809::serial::usart2::instance() }}
}
#[cfg(feature="panicout_usart3")]
pub mod panicout {
  #[doc(hidden)]
  #[macro_export]
  macro_rules! panic_stdout { () => { avr_oxide::hal::atmega4809::serial::usart3::instance() }}
}

// Code ======================================================================
#[cfg(feature="usart0")]
pub mod usart0 {
  use avr_oxide::{atmel_0series_usart_tpl};
  use avr_oxide::hal::generic::serial::base::zeroseries::MuxControl;
  #[cfg(target_arch="avr")]
  use avr_oxide::hal::atmega4809::cpu::PortmuxControl;

  pub enum SerialPortPins {
    PortA30,
    PortA74,
    NotConnected
  }



  impl MuxControl for SerialPortPins {
    #[cfg(target_arch="avr")]
    fn route(&self) {
      unsafe {
        let pmux : &mut PortmuxControl = core::mem::transmute(super::super::ADDR_PORTMUX);
        let routectl = &mut pmux.usartroutea as *mut u8;


        let current = core::ptr::read_volatile(routectl);
        let mask    = 0b00000011;
        core::ptr::write_volatile(routectl,
                                  (current & !mask) | match self {
                                    SerialPortPins::PortA30 => 0b000000_00,
                                    SerialPortPins::PortA74 => 0b000000_01,
                                    SerialPortPins::NotConnected => 0b000000_11
                                  });
      }
    }

    #[cfg(not(target_arch="avr"))]
    fn route(&self) {
      println!("*** USART:PORTMUX: Routed portmux");
    }
  }

  atmel_0series_usart_tpl!(super::super::ADDR_USART0,
    avr_oxide::hal::generic::serial::SerialPortIdentity::Usart0,
    avr_oxide::hal::generic::serial::base::zeroseries::AtmelUsart<SerialPortPins,64>,
    _ivr_usart0_rcx, _ivr_usart0_tcx, _ivr_usart0_dre);


}

#[cfg(feature="usart1")]
pub mod usart1 {
  use avr_oxide::{atmel_0series_usart_tpl};
  use avr_oxide::hal::generic::serial::base::zeroseries::MuxControl;
  #[cfg(target_arch="avr")]
  use avr_oxide::hal::atmega4809::cpu::PortmuxControl;

  pub enum SerialPortPins {
    PortC30,
    PortC74,
    NotConnected
  }
  impl MuxControl for SerialPortPins {
    #[cfg(target_arch="avr")]
    fn route(&self) {
      unsafe {
        let pmux : &mut PortmuxControl = core::mem::transmute(super::super::ADDR_PORTMUX);
        let routectl = &mut pmux.usartroutea as *mut u8;

        let current = core::ptr::read_volatile(routectl);
        let mask    = 0b00001100;
        core::ptr::write_volatile(routectl,
                                  (current & !mask) | match self {
                                    SerialPortPins::PortC30 => 0b0000_00_00,
                                    SerialPortPins::PortC74 => 0b0000_01_00,
                                    SerialPortPins::NotConnected => 0b0000_11_00
                                  });
      }
    }

    #[cfg(not(target_arch="avr"))]
    fn route(&self) {
      println!("*** USART:PORTMUX: Routed portmux");
    }
  }

  atmel_0series_usart_tpl!(super::super::ADDR_USART1,
    avr_oxide::hal::generic::serial::SerialPortIdentity::Usart1,
    avr_oxide::hal::generic::serial::base::zeroseries::AtmelUsart<SerialPortPins,64>,
    _ivr_usart1_rcx, _ivr_usart1_tcx, _ivr_usart1_dre);


}

#[cfg(feature="usart2")]
pub mod usart2 {
  use avr_oxide::{atmel_0series_usart_tpl};
  use avr_oxide::hal::generic::serial::base::zeroseries::MuxControl;
  #[cfg(target_arch="avr")]
  use avr_oxide::hal::atmega4809::cpu::PortmuxControl;

  pub enum SerialPortPins {
    PortF30,
    PortF64,
    NotConnected
  }
  impl MuxControl for SerialPortPins {
    #[cfg(target_arch="avr")]
    fn route(&self) {
      unsafe {
        let pmux : &mut PortmuxControl = core::mem::transmute(super::super::ADDR_PORTMUX);
        let routectl = &mut pmux.usartroutea as *mut u8;

        let current = core::ptr::read_volatile(routectl);
        let mask    = 0b00110000;
        core::ptr::write_volatile(routectl,
                                  (current & !mask) | match self {
                                    SerialPortPins::PortF30 => 0b00_00_0000,
                                    SerialPortPins::PortF64 => 0b00_01_0000,
                                    SerialPortPins::NotConnected => 0b00_11_0000
                                  });
      }
    }

    #[cfg(not(target_arch="avr"))]
    fn route(&self) {
      println!("*** USART:PORTMUX: Routed portmux");
    }
  }

  atmel_0series_usart_tpl!(super::super::ADDR_USART2,
    avr_oxide::hal::generic::serial::SerialPortIdentity::Usart2,
    avr_oxide::hal::generic::serial::base::zeroseries::AtmelUsart<SerialPortPins, 64>,
    _ivr_usart2_rcx, _ivr_usart2_tcx, _ivr_usart2_dre);


}

#[cfg(feature="usart3")]
pub mod usart3 {
  use avr_oxide::{atmel_0series_usart_tpl};
  use avr_oxide::hal::generic::serial::base::zeroseries::MuxControl;
  #[cfg(target_arch="avr")]
  use avr_oxide::hal::atmega4809::cpu::PortmuxControl;

  pub enum SerialPortPins {
    PortB30,
    PortB54,
    NotConnected
  }
  impl MuxControl for SerialPortPins {
    #[cfg(target_arch="avr")]
    #[allow(clippy::unusual_byte_groupings)]
    fn route(&self) {
      unsafe {
        let pmux : &mut PortmuxControl = core::mem::transmute(super::super::ADDR_PORTMUX);
        let routectl = &mut pmux.usartroutea as *mut u8;

        let current = core::ptr::read_volatile(routectl);
        let mask    = 0b11000000;
        core::ptr::write_volatile(routectl,
                                  (current & !mask) | match self {
                                    SerialPortPins::PortB30 => 0b00_000000,
                                    SerialPortPins::PortB54 => 0b01_000000,
                                    SerialPortPins::NotConnected => 0b11_000000
                                  });
      }
    }

    #[cfg(not(target_arch="avr"))]
    fn route(&self) {
      println!("*** USART:PORTMUX: Routed portmux");
    }
  }

  atmel_0series_usart_tpl!(super::super::ADDR_USART3,
    avr_oxide::hal::generic::serial::SerialPortIdentity::Usart3,
    avr_oxide::hal::generic::serial::base::zeroseries::AtmelUsart<SerialPortPins, 64>,
    _ivr_usart3_rcx, _ivr_usart3_tcx, _ivr_usart3_dre);
}
