/* mod.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! ATmega328P implementations of the generic device traits
#![allow(unused)]
use avr_oxide::hal::generic::debugled::DebugLed;
use avr_oxide::hal::generic::serial::SerialNoInterruptTx;

pub mod cpu;
pub mod port;
pub mod hardware;
pub mod debugled;
pub mod serial;


#[cfg(feature="panic_handler")]
#[panic_handler]
#[cfg(target_arch="avr")]
fn panic(info: &core::panic::PanicInfo) -> ! {
  #![allow(deprecated)] // until asm! is implemented for AVR
  unsafe {
    avr_oxide::hal::concurrency::interrupt::disable_interrupts();

    #[cfg(feature="panicout")]
    {
      use avr_oxide::util::{debug_print, debug_print_u16};
      // This doesn't look too pretty, but we can't use any of the standard
      // formatting routines because they require an allocator, which even
      // if we have one, is probably broken right now.
      // Plus the standard formatter routines don't know that we can't use
      // 32 bit arithmetic at the moment 😖
      debug_print("\n\nPanic:\n");
      if let Some(location) = info.location() {
        debug_print("   --> ");
        debug_print(&location.file());
        debug_print(":");
        debug_print_u16(location.line() as u16);
        debug_print(":");
        debug_print_u16(location.column() as u16);
        debug_print("\n");
      }
    }


    llvm_asm!("break" :::: "volatile");

    loop {
      #[cfg(feature="arduino")]
      avr_oxide::hal::atmega328p::debugled::PinB5DebugLed::toggle();

      for _i in 0..255 {
        avr_oxide::hal::generic::busy_loop(255);
      }
    }
  }
}

const ADDR_CPU:    u16 = 0x005D;

const ADDR_PORTB:  u16 = 0x0023;
const ADDR_PORTC:  u16 = 0x0026;
const ADDR_PORTD:  u16 = 0x0029;

const ADDR_USART:  u16 = 0x00C0;
const ADDR_CLKPR:  u16 = 0x0061;

const ADDR_PCICR:  u16 = 0x0068;
const ADDR_PCIFR:  u16 = 0x003B;
const ADDR_PCMSK0: u16 = 0x006B;
const ADDR_PCMSK1: u16 = 0x006C;
const ADDR_PCMSK2: u16 = 0x006D;