/* hardware.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Provides a single point of access to the hardware devices associated with
//! the AVR CPU on which we are running.
//!
//! This struct is your first port of call for accessing a pin, timer, or
//! any other hardware component which we have support for.
//!
//! It is implemented as a global singleton, a reference is obtained using
//! the provided instance() function.  From here, you can then obtain
//! references to all the supported low-level hardware devices.
//!
//! # Example usage
//! ```rust
//! use avr_oxide::hal::atmega4809;
//!
//! fn use_hardware() {
//!   let hardware = atmega328p::hardware::instance();
//!
//!   // Get a reference to Port D
//!   let portd = &hardware.portd;
//! }
//! ```

// Imports ===================================1================================
use avr_oxide::hal::atmega328p;
use avr_oxide::mut_singleton;

// Declarations ==============================================================
pub struct Hardware {
  pub portb: &'static atmega328p::port::portb::PortImpl,
  pub portc: &'static atmega328p::port::portc::PortImpl,
  pub portd: &'static atmega328p::port::portd::PortImpl,

  #[cfg(feature="usart0")]
  pub usart0: &'static mut atmega328p::serial::usart0::SerialImpl,
}

// Code ======================================================================
mut_singleton!(Hardware, HARDWARE, instance,
  Hardware {
    portb: atmega328p::port::portb::instance(),
    portc: atmega328p::port::portc::instance(),
    portd: atmega328p::port::portd::instance(),

    #[cfg(feature="usart0")]
    usart0: atmega328p::serial::usart0::instance(),
  }
);
