/* led.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Simple abstraction of an LED attached to a GPIO pin.
//!
//! In all honestly, it doesn't abstract much ;).  But it does allow a
//! consistent method for using/creating by implementing the [`avr_oxide::devices::UsesPin`] trait,
//! and being compatible with the Handle trait.
//!
//! # Usage
//! ```no_run
//! # #![no_std]
//! # #![no_main]
//! #
//! # use avr_oxide::hal::atmega4809::hardware;
//! # use avr_oxide::devices::UsesPin;
//! # use avr_oxide::devices::OxideLed;
//! # use avr_oxide::arduino;
//!
//! # #[avr_oxide::main(chip="atmega4809")]
//! # pub fn main() {
//! #   let supervisor = avr_oxide::oxide::instance();
//! #   let hardware = hardware::instance();
//!   let arduino = arduino::nanoevery::Arduino::from(hardware);
//!
//!   let green_led = OxideLed::with_pin(arduino.d7);
//!
//!   green_led.toggle();
//! #
//! #  supervisor.run();
//! # }
//! ```


// Imports ===================================================================
use avr_oxide::hal::generic::port::{Pin, PinMode, InterruptMode};
use avr_oxide::util::OwnOrBorrowMut;
use avr_oxide::devices::UsesPin;
use avr_oxide::devices::internal::StaticShareable;


// Declarations ==============================================================
pub struct Led<P>
where
  P: 'static + Pin
{
  pin: OwnOrBorrowMut<'static,P>
}

impl<P> StaticShareable for Led<P>
where
  P: 'static + Pin
{}

// Code ======================================================================
impl<P> UsesPin<P> for Led<P>
where
  P: Pin
{
  fn using<OP: Into<OwnOrBorrowMut<'static, P>>>(pin: OP) -> Self {
    let pin : OwnOrBorrowMut<P> = pin.into();

    pin.set_mode(PinMode::Output);
    pin.set_interrupt_mode(InterruptMode::Disabled);
    Led { pin }
  }
}

impl<P> Led<P>
where
  P: Pin
{
  pub fn set_on(&self) {
    self.pin.set_high();
  }

  pub fn set_off(&self) {
    self.pin.set_low();
  }

  pub fn set(&self, on: bool) {
    match on {
      true => self.set_on(),
      false => self.set_off()
    }
  }

  pub fn toggle(&self) {
    self.pin.toggle()
  }
}
