/* deviceconsts.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Device-specific constants.  These are determined by the feature flags
//! configued in your `Cargo.toml`.
//!
//! # Features
//! The following feature flags affect these constants
//!
//! | Type | Values |
//! | ---- | ------ |
//! | CPU identification | `atmega4809` |
//! | Clockspeed | `20MHz`, `16MHz` |
//! | Allocator size | `alloc_small`, `alloc_medium`, `alloc_large` |
//!

// Imports ===================================================================
// Declarations ==============================================================
#[cfg(not(any(feature="20MHz", feature="16MHz")))]
compile_error!("A clockspeed feature (\"16MHz\", \"20MHz\") must be enabled for this crate");

#[cfg(not(any(feature="atmega4809", feature="atmega328p")))]
compile_error!("A processor feature (\"atmega4809\", \"atmega328p\") must be enabled for this crate");

#[cfg(not(any(feature="alloc_small",feature="alloc_medium",feature="alloc_large")))]
compile_error!("An allocator feature (\"alloc_small\", \"alloc_medium\", \"alloc_large\") must be enabled for this crate");

#[cfg(feature = "atmega4809")]
pub mod memory {
  pub const FLASH_SIZE  : usize  = 49_152;
  pub const SRAM_SIZE   : usize  =  6_144;
  pub const EEPROM_SIZE : usize  =    256;
  pub const USER_SIZE   : usize  =     64;

  #[cfg(feature = "alloc_small")]
  pub mod alloc {
    pub        const HEAPSIZE : usize = 512;
    pub(crate) const SMAX     : usize = 8;
  }
  #[cfg(feature = "alloc_medium")]
  pub mod alloc {
    pub        const HEAPSIZE : usize = 1024;
    pub(crate) const SMAX     : usize = 16;
  }
  #[cfg(feature = "alloc_large")]
  pub mod alloc {
    pub        const HEAPSIZE : usize = 3072;
    pub(crate) const SMAX     : usize = 32;
  }

  pub mod eeprom {
    pub const EEPROM_SIZE      : usize = 256;
    pub const USERROW_SIZE     : usize = 64;
  }
}

#[cfg(feature = "atmega328p")]
pub mod memory {
  pub const FLASH_SIZE  : usize  = 32_768;
  pub const SRAM_SIZE   : usize  =  2_048;
  pub const EEPROM_SIZE : usize  =  1_024;

  #[cfg(feature = "alloc_small")]
  pub mod alloc {
    pub        const HEAPSIZE : usize = 128;
    pub(crate) const SMAX     : usize = 8;
  }
  #[cfg(feature = "alloc_medium")]
  pub mod alloc {
    pub        const HEAPSIZE : usize = 256;
    pub(crate) const SMAX     : usize = 8;
  }
  #[cfg(feature = "alloc_large")]
  pub mod alloc {
    pub        const HEAPSIZE : usize = 512;
    pub(crate) const SMAX     : usize = 8;
  }

  pub mod eeprom {
    pub const EEPROM_SIZE      : usize = 1024;
  }
}

pub mod oxide {
  pub const MAX_THREADS : usize  = 1;
}

#[cfg(feature="20MHz")]
pub mod clock {
  pub const MASTER_CLOCK_HZ: u32         = 20_000_000; // Config'd in device fuses

  #[cfg(feature="power_low")]
  pub const MASTER_CLOCK_PRESCALER:  u8  = 8;
  #[cfg(feature="power_low")]
  pub const MASTER_TICK_FREQ_HZ: u16     = 500;

  #[cfg(feature="power_med")]
  pub const MASTER_CLOCK_PRESCALER:  u8  = 4;
  #[cfg(feature="power_med")]
  pub const MASTER_TICK_FREQ_HZ: u16     = 500;

  #[cfg(not(any(feature="power_med",feature="power_low")))]
  pub const MASTER_CLOCK_PRESCALER:  u8  = 1;
  #[cfg(not(any(feature="power_med",feature="power_low")))]
  pub const MASTER_TICK_FREQ_HZ: u16     = 1000;

}

#[cfg(feature="16MHz")]
pub mod clock {
  pub const MASTER_CLOCK_HZ: u32         = 16_000_000; // Config'd in device fuses

  #[cfg(feature="power_low")]
  pub const MASTER_CLOCK_PRESCALER:  u8  = 8;
  #[cfg(feature="power_low")]
  pub const MASTER_TICK_FREQ_HZ: u16     = 500;

  #[cfg(feature="power_med")]
  pub const MASTER_CLOCK_PRESCALER:  u8  = 4;
  #[cfg(feature="power_med")]
  pub const MASTER_TICK_FREQ_HZ: u16     = 500;

  #[cfg(not(any(feature="power_med",feature="power_low")))]
  pub const MASTER_CLOCK_PRESCALER:  u8  = 1;
  #[cfg(not(any(feature="power_med",feature="power_low")))]
  pub const MASTER_TICK_FREQ_HZ: u16     = 1000;
}


// Code ======================================================================

// Tests =====================================================================
