/* boot.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Boot process for the Oxide operating system.  Initialises memory and
//! interrupt vectors, loads our hardware table, and then calls the
//! oxide_main() method provided by whoever is using our crate.

// Imports ===================================================================
use avr_oxide::hal::concurrency;
use avr_oxide::hal::concurrency::thread;
use avr_oxide::hal::generic::cpu::{ClockControl, SleepControl};
#[cfg(feature="alloc_ftr")]
use avr_oxide::alloc;
use avr_oxide::deviceconsts::clock;
use avr_oxide::sleepctrl;

// Declarations ==============================================================
extern {
  fn __oxide_main() -> !;
}

// Code ======================================================================
#[no_mangle]
fn _oxide_boot() {
  unsafe {
    // If we have a global allocator, initialise it
    #[cfg(feature="alloc_ftr")]
    alloc::initialise();

    // Inititalise the threading/concurrency system
    concurrency::internal::initialise();

    // Reset the sleep mode
    sleepctrl!().reset();

    // Set up any hardware-specific configuration
    #[cfg(feature="atmega4809")]
    avr_oxide::hal::atmega4809::cpu::clock().clk_per_prescaler(clock::MASTER_CLOCK_PRESCALER);
    #[cfg(feature="atmega328p")]
    avr_oxide::hal::atmega328p::cpu::clock().clk_per_prescaler(clock::MASTER_CLOCK_PRESCALER);


    // Create the supervisor
    let mut supervisor = avr_oxide::oxide::create_supervisor();
    avr_oxide::oxide::set_global_supervisor(&mut supervisor);

    // Spawn the main thread
    let _jh = thread::spawn_threadunsafe(||{
      __oxide_main();
    });

    // Enter the scheduler loop
    concurrency::internal::run_scheduler();
  }
}
