/* arduino.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! A helper module that maps standard ATmega pins into equivalent Arduino
//! names.
//!
//! # Usage
//! You can create an instance of the `Arduino` type From the generic
//! hardware instance provided by the `avr_oxide::hal::<<device>>::hardware`
//! crate.  This then allows you to refer to pins etc. using the Arduino
//! naming conventions instead of the AVR device's own naming conventions.
//!
//! ```rust
//! use avr_oxide::hal::atmega4809;
//! use avr_oxide::arduino;
//! use avr_oxide::hal::generic::port::{ Pin, PinMode };
//!
//! fn use_arduino_types() {
//!   let hardware = atmega4809::hardware::instance();
//!   let arduino  = arduino::nanoevery::Arduino::from(hardware);
//!
//!   // We can now refer to Pord D, pin 3 on an Arduino Nano Every as
//!   // `arduino.a0`
//!   arduino.a0.set_mode(PinMode::Output);
//!   arduino.a0.set_high();
//! }
//! ```

// Imports ===================================================================

// Declarations ==============================================================
#[cfg(feature="atmega4809")]
pub mod nanoevery {
  use avr_oxide::hal::atmega4809::hardware::Hardware;
  use avr_oxide::hal::atmega4809;
  use avr_oxide::hal::generic::port::PinIdentity;

  /**
   * Arduino-style pin identifier, used to identify a pin event source at
   * runtime
   */
  pub enum ArduinoPinIdentity {
    /// The so-called 'analogue' pins (that is, pins that can be attached to
    /// a timer's PWM output)
    PinA(u8),
    /// The 'digital' pins
    PinD(u8)
  }

  pub mod pintypes {
    pub type A0  = avr_oxide::hal::atmega4809::port::portd::PinImpl;
    pub type A1  = avr_oxide::hal::atmega4809::port::portd::PinImpl;
    pub type A2  = avr_oxide::hal::atmega4809::port::portd::PinImpl;
    pub type A3  = avr_oxide::hal::atmega4809::port::portd::PinImpl;
    pub type A4  = avr_oxide::hal::atmega4809::port::porta::PinImpl;
    pub type A5  = avr_oxide::hal::atmega4809::port::porta::PinImpl;
    pub type A6  = avr_oxide::hal::atmega4809::port::portd::PinImpl;
    pub type A7  = avr_oxide::hal::atmega4809::port::portd::PinImpl;

    pub type D0  = avr_oxide::hal::atmega4809::port::portc::PinImpl;
    pub type D1  = avr_oxide::hal::atmega4809::port::portc::PinImpl;
    pub type D2  = avr_oxide::hal::atmega4809::port::porta::PinImpl;
    pub type D3  = avr_oxide::hal::atmega4809::port::portf::PinImpl;
    pub type D4  = avr_oxide::hal::atmega4809::port::portc::PinImpl;
    pub type D5  = avr_oxide::hal::atmega4809::port::portb::PinImpl;
    pub type D6  = avr_oxide::hal::atmega4809::port::portf::PinImpl;
    pub type D7  = avr_oxide::hal::atmega4809::port::porta::PinImpl;
    pub type D8  = avr_oxide::hal::atmega4809::port::porte::PinImpl;
    pub type D9  = avr_oxide::hal::atmega4809::port::portb::PinImpl;
    pub type D10 = avr_oxide::hal::atmega4809::port::portb::PinImpl;
    pub type D11 = avr_oxide::hal::atmega4809::port::porte::PinImpl;
    pub type D12 = avr_oxide::hal::atmega4809::port::porte::PinImpl;
    pub type D13 = avr_oxide::hal::atmega4809::port::porte::PinImpl;
  }

  pub struct Arduino {
    #[cfg(feature="tcb0")]
    pub timer0: &'static mut atmega4809::timer::tcb0::TimerImpl,
    #[cfg(feature="tcb1")]
    pub timer1: &'static mut atmega4809::timer::tcb1::TimerImpl,
    #[cfg(feature="tcb2")]
    pub timer2: &'static mut atmega4809::timer::tcb2::TimerImpl,
    #[cfg(feature="tcb3")]
    pub timer3: &'static mut atmega4809::timer::tcb3::TimerImpl,
    #[cfg(feature="rtc")]
    pub rtc: &'static mut atmega4809::timer::rtc::TimerImpl,

    pub a0: &'static mut pintypes::A0,
    pub a1: &'static mut pintypes::A1,
    pub a2: &'static mut pintypes::A2,
    pub a3: &'static mut pintypes::A3,
    pub a4: &'static mut pintypes::A4,
    pub a5: &'static mut pintypes::A5,
    pub a6: &'static mut pintypes::A6,
    pub a7: &'static mut pintypes::A7,

    pub d0: &'static mut pintypes::D0,
    pub d1: &'static mut pintypes::D1,
    pub d2: &'static mut pintypes::D2,
    pub d3: &'static mut pintypes::D3,
    pub d4: &'static mut pintypes::D4,
    pub d5: &'static mut pintypes::D5,
    pub d6: &'static mut pintypes::D6,
    pub d7: &'static mut pintypes::D7,
    pub d8: &'static mut pintypes::D8,
    pub d9: &'static mut pintypes::D9,
    pub d10: &'static mut pintypes::D10,
    pub d11: &'static mut pintypes::D11,
    pub d12: &'static mut pintypes::D12,
    pub d13: &'static mut pintypes::D13,

    pub watchdog: &'static mut atmega4809::watchdog::WatchdogImpl,

    pub eeprom: &'static mut atmega4809::eeprom::eeprom::EepromImpl,
    pub userrow: &'static mut atmega4809::eeprom::userrow::EepromImpl,

    #[cfg(feature="usart3")]
    pub usb_serial: &'static mut atmega4809::serial::usart3::SerialImpl,
    #[cfg(feature="usart3")]
    pub usb_serial_tx: &'static mut atmega4809::port::portb::PinImpl,
    #[cfg(feature="usart3")]
    pub usb_serial_rx: &'static mut atmega4809::port::portb::PinImpl,
  }


  impl From<&'static mut avr_oxide::hal::atmega4809::hardware::Hardware> for Arduino {
    fn from(avr: &'static mut Hardware) -> Self {
      Arduino {
        #[cfg(feature="tcb0")]
        timer0: avr.timerb0,
        #[cfg(feature="tcb1")]
        timer1: avr.timerb1,
        #[cfg(feature="tcb2")]
        timer2: avr.timerb2,
        #[cfg(feature="tcb3")]
        timer3: avr.timerb3,
        #[cfg(feature="rtc")]
        rtc: avr.rtc,

        a0: atmega4809::port::portd::pin_instance(3),
        a1: atmega4809::port::portd::pin_instance(2),
        a2: atmega4809::port::portd::pin_instance(1),
        a3: atmega4809::port::portd::pin_instance(0),
        a4: atmega4809::port::porta::pin_instance(2),
        a5: atmega4809::port::porta::pin_instance(3),
        a6: atmega4809::port::portd::pin_instance(4),
        a7: atmega4809::port::portd::pin_instance(5),
        d0: atmega4809::port::portc::pin_instance(4),
        d1: atmega4809::port::portc::pin_instance(5),
        d2: atmega4809::port::porta::pin_instance(0),
        d3: atmega4809::port::portf::pin_instance(5),
        d4: atmega4809::port::portc::pin_instance(6),
        d5: atmega4809::port::portb::pin_instance(2),
        d6: atmega4809::port::portf::pin_instance(4),
        d7: atmega4809::port::porta::pin_instance(1),
        d8: atmega4809::port::porte::pin_instance(3),
        d9: atmega4809::port::portb::pin_instance(0),
        d10: atmega4809::port::portb::pin_instance(1),
        d11: atmega4809::port::porte::pin_instance(0),
        d12: atmega4809::port::porte::pin_instance(1),
        d13: atmega4809::port::porte::pin_instance(2),

        watchdog: avr.watchdog,

        eeprom: avr.eeprom,
        userrow: avr.userrow,

        #[cfg(feature="usart3")]
        usb_serial: avr.usart3.mux(atmega4809::serial::usart3::SerialPortPins::PortB54),
        #[cfg(feature="usart3")]
        usb_serial_tx: atmega4809::port::portb::pin_instance(4),
        #[cfg(feature="usart3")]
        usb_serial_rx: atmega4809::port::portb::pin_instance(5),
      }
    }
  }

  impl From<PinIdentity> for ArduinoPinIdentity {
    fn from(id: PinIdentity) -> Self {
      match id {
        PinIdentity::PortA(0) => ArduinoPinIdentity::PinD(2),
        PinIdentity::PortA(1) => ArduinoPinIdentity::PinD(7),
        PinIdentity::PortA(2) => ArduinoPinIdentity::PinA(4),
        PinIdentity::PortA(3) => ArduinoPinIdentity::PinA(5),

        PinIdentity::PortB(0) => ArduinoPinIdentity::PinD(9),
        PinIdentity::PortB(1) => ArduinoPinIdentity::PinD(10),
        PinIdentity::PortB(2) => ArduinoPinIdentity::PinD(5),

        PinIdentity::PortC(4) => ArduinoPinIdentity::PinD(0),
        PinIdentity::PortC(5) => ArduinoPinIdentity::PinD(1),
        PinIdentity::PortC(6) => ArduinoPinIdentity::PinD(4),

        PinIdentity::PortD(0) => ArduinoPinIdentity::PinA(3),
        PinIdentity::PortD(1) => ArduinoPinIdentity::PinA(2),
        PinIdentity::PortD(2) => ArduinoPinIdentity::PinA(1),
        PinIdentity::PortD(3) => ArduinoPinIdentity::PinA(0),
        PinIdentity::PortD(4) => ArduinoPinIdentity::PinA(6),
        PinIdentity::PortD(5) => ArduinoPinIdentity::PinA(7),

        PinIdentity::PortE(3) => ArduinoPinIdentity::PinD(8),
        PinIdentity::PortE(0) => ArduinoPinIdentity::PinD(11),
        PinIdentity::PortE(1) => ArduinoPinIdentity::PinD(12),
        PinIdentity::PortE(2) => ArduinoPinIdentity::PinD(13),

        PinIdentity::PortF(5) => ArduinoPinIdentity::PinD(3),
        PinIdentity::PortF(4) => ArduinoPinIdentity::PinD(6),

        _ => panic!()
      }
    }
  }
}

#[cfg(feature="atmega328p")]
pub mod uno {
  use avr_oxide::hal::atmega328p::hardware::Hardware;
  use avr_oxide::hal::atmega328p;
  use avr_oxide::hal::generic::port::PinIdentity;

  /**
   * Arduino-style pin identifier, used to identify a pin event source at
   * runtime
   */
  pub enum ArduinoPinIdentity {
    /// The so-called 'analogue' pins (that is, pins that can be attached to
    /// a timer's PWM output)
    PinA(u8),
    /// The 'digital' pins
    PinD(u8)
  }

  pub mod pintypes {
    pub type A0  = avr_oxide::hal::atmega328p::port::portc::PinImpl;
    pub type A1  = avr_oxide::hal::atmega328p::port::portc::PinImpl;
    pub type A2  = avr_oxide::hal::atmega328p::port::portc::PinImpl;
    pub type A3  = avr_oxide::hal::atmega328p::port::portc::PinImpl;
    pub type A4  = avr_oxide::hal::atmega328p::port::portc::PinImpl;
    pub type A5  = avr_oxide::hal::atmega328p::port::portc::PinImpl;

    pub type D0  = avr_oxide::hal::atmega328p::port::portd::PinImpl;
    pub type D1  = avr_oxide::hal::atmega328p::port::portd::PinImpl;
    pub type D2  = avr_oxide::hal::atmega328p::port::portd::PinImpl;
    pub type D3  = avr_oxide::hal::atmega328p::port::portd::PinImpl;
    pub type D4  = avr_oxide::hal::atmega328p::port::portd::PinImpl;
    pub type D5  = avr_oxide::hal::atmega328p::port::portd::PinImpl;
    pub type D6  = avr_oxide::hal::atmega328p::port::portd::PinImpl;
    pub type D7  = avr_oxide::hal::atmega328p::port::portd::PinImpl;
    pub type D8  = avr_oxide::hal::atmega328p::port::portb::PinImpl;
    pub type D9  = avr_oxide::hal::atmega328p::port::portb::PinImpl;
    pub type D10 = avr_oxide::hal::atmega328p::port::portb::PinImpl;
    pub type D11 = avr_oxide::hal::atmega328p::port::portb::PinImpl;
    pub type D12 = avr_oxide::hal::atmega328p::port::portb::PinImpl;
    pub type D13 = avr_oxide::hal::atmega328p::port::portb::PinImpl;
  }

  pub struct Arduino {
    pub a0: &'static mut pintypes::A0,
    pub a1: &'static mut pintypes::A1,
    pub a2: &'static mut pintypes::A2,
    pub a3: &'static mut pintypes::A3,
    pub a4: &'static mut pintypes::A4,
    pub a5: &'static mut pintypes::A5,

    pub d0: &'static mut pintypes::D0,
    pub d1: &'static mut pintypes::D1,
    pub d2: &'static mut pintypes::D2,
    pub d3: &'static mut pintypes::D3,
    pub d4: &'static mut pintypes::D4,
    pub d5: &'static mut pintypes::D5,
    pub d6: &'static mut pintypes::D6,
    pub d7: &'static mut pintypes::D7,
    pub d8: &'static mut pintypes::D8,
    pub d9: &'static mut pintypes::D9,
    pub d10: &'static mut pintypes::D10,
    pub d11: &'static mut pintypes::D11,
    pub d12: &'static mut pintypes::D12,
    pub d13: &'static mut pintypes::D13,

    #[cfg(feature="usart0")]
    pub usb_serial: &'static mut atmega328p::serial::usart0::SerialImpl,
    #[cfg(feature="usart0")]
    pub usb_serial_tx: &'static mut atmega328p::port::portd::PinImpl,
    #[cfg(feature="usart0")]
    pub usb_serial_rx: &'static mut atmega328p::port::portd::PinImpl,
  }


  impl From<&'static mut avr_oxide::hal::atmega328p::hardware::Hardware> for Arduino {
    fn from(avr: &'static mut Hardware) -> Self {
      Arduino {

        a0: atmega328p::port::portc::pin_instance(0),
        a1: atmega328p::port::portc::pin_instance(1),
        a2: atmega328p::port::portc::pin_instance(2),
        a3: atmega328p::port::portc::pin_instance(3),
        a4: atmega328p::port::portc::pin_instance(4),
        a5: atmega328p::port::portc::pin_instance(5),

        d0: atmega328p::port::portd::pin_instance(0),
        d1: atmega328p::port::portd::pin_instance(1),
        d2: atmega328p::port::portd::pin_instance(2),
        d3: atmega328p::port::portd::pin_instance(3),
        d4: atmega328p::port::portd::pin_instance(4),
        d5: atmega328p::port::portd::pin_instance(5),
        d6: atmega328p::port::portd::pin_instance(6),
        d7: atmega328p::port::portd::pin_instance(7),

        d8: atmega328p::port::portb::pin_instance(0),
        d9: atmega328p::port::portb::pin_instance(1),
        d10: atmega328p::port::portb::pin_instance(2),
        d11: atmega328p::port::portb::pin_instance(3),
        d12: atmega328p::port::portb::pin_instance(4),
        d13: atmega328p::port::portb::pin_instance(5),

        #[cfg(feature="usart0")]
        usb_serial: avr.usart0,
        #[cfg(feature="usart0")]
        usb_serial_tx: atmega328p::port::portd::pin_instance(1),
        #[cfg(feature="usart0")]
        usb_serial_rx: atmega328p::port::portd::pin_instance(0),
      }
    }
  }

  impl From<PinIdentity> for ArduinoPinIdentity {
    fn from(id: PinIdentity) -> Self {
      match id {
        PinIdentity::PortD(0) => ArduinoPinIdentity::PinD(0),
        PinIdentity::PortD(1) => ArduinoPinIdentity::PinD(1),
        PinIdentity::PortD(2) => ArduinoPinIdentity::PinD(2),
        PinIdentity::PortD(3) => ArduinoPinIdentity::PinD(3),
        PinIdentity::PortD(4) => ArduinoPinIdentity::PinD(4),
        PinIdentity::PortD(5) => ArduinoPinIdentity::PinD(5),
        PinIdentity::PortD(6) => ArduinoPinIdentity::PinD(6),
        PinIdentity::PortD(7) => ArduinoPinIdentity::PinD(7),

        PinIdentity::PortB(0) => ArduinoPinIdentity::PinD(8),
        PinIdentity::PortB(1) => ArduinoPinIdentity::PinD(9),
        PinIdentity::PortB(2) => ArduinoPinIdentity::PinD(10),
        PinIdentity::PortB(3) => ArduinoPinIdentity::PinD(11),
        PinIdentity::PortB(4) => ArduinoPinIdentity::PinD(12),
        PinIdentity::PortB(5) => ArduinoPinIdentity::PinD(13),

        PinIdentity::PortC(0) => ArduinoPinIdentity::PinA(0),
        PinIdentity::PortC(1) => ArduinoPinIdentity::PinA(1),
        PinIdentity::PortC(2) => ArduinoPinIdentity::PinA(2),
        PinIdentity::PortC(3) => ArduinoPinIdentity::PinA(3),
        PinIdentity::PortC(4) => ArduinoPinIdentity::PinA(4),
        PinIdentity::PortC(5) => ArduinoPinIdentity::PinA(5),

        _ => panic!()
      }
    }
  }
}
// Tests =====================================================================
