/* serial.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Generic AVR serial driver (USARTs, SPI, TWI, etc.) support

// Imports ===================================================================

// Declarations ==============================================================

use core::any::Any;
use ufmt::derive::uDebug;
use oxide_macros::Persist;
use avr_oxide::hal::generic::callback::IsrCallback;

/**
 * Identifies a particular source of serial port events, i.e. a particular
 * hardware USART, at runtime.
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum SerialPortIdentity {
  /// Universial Synchronous/Asynchronous Receiver/Transmitter 0
  Usart0,
  /// Universial Synchronous/Asynchronous Receiver/Transmitter 1
  Usart1,
  /// Universial Synchronous/Asynchronous Receiver/Transmitter 2
  Usart2,
  /// Universial Synchronous/Asynchronous Receiver/Transmitter 3
  Usart3
}

/**
 * Possible baud rates we can configure a serial port clock to use.
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum BaudRate {
  /// 300 baud
  Baud300,
  /// 600 baud
  Baud600,
  /// 1200 baud
  Baud1200,
  /// 2400 baud
  Baud2400,
  /// 4800 baud
  Baud4800,
  /// 9600 baud
  Baud9600,
  /// 14400 baud
  Baud14400,
  /// 19200 baud
  Baud19200,
  /// 28800 baud
  Baud28800,
  /// 38400 baud
  Baud38400,
  /// 57600 baud
  Baud57600,
  // 76800 baud.  Generating this baud rate with reasonable accuracy depends
  // on a CPU clockrate of 16MHz or above.
  Baud76800,
  // 117200 baud.  Generating this baud rate with reasonable accuracy
  // depends on a CPU clockrate of 16MHz or above.
  Baud115200,
  // Not going higher than this because the chances of generating accurate
  // higher rates from the peripheral clock on our controllers at their typical
  // clockrates is basically zero.

  /// Automatic baud rate detection.  See the AVR manual, but don't expect
  /// me to test whether or not this actually works ;-)
  Auto
}
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum SynchronousMode {
  /// Master mode - I will generate the clock for everyone
  Master(BaudRate),
  /// Slave, I will - use the clock generated by the master
  Slave
}

/**
 * The number of data bits in each character frame
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum DataBits {
  /// 5 bit data frame
  Bits5,
  /// 6 bit data frame
  Bits6,
  /// 7 bit data frame
  Bits7,
  /// 8 bit data frame
  Bits8,
  /// 9 bit data frame, little-endian (low byte sent first)
  Bits9LE,
  /// 9 bit data frame, big-endian (high byte sent first)
  Bits9HE
}

/**
 * Parity for each transmitted character
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum Parity {
  None,
  Even,
  Odd
}

/**
 * The number of stop bits (wait states effectively) to transmit after each
 * character frame
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum StopBits {
  Bits1,
  Bits2
}

/**
 * The serial communications mode for a serial port peripheral.
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum SerialPortMode {
  /// Asynchronous communication
  Asynch(BaudRate,DataBits,Parity,StopBits),
  /// Synchronous communication
  Synch(SynchronousMode)
}

/**
 * Serial communication errors
 */
#[derive(Clone,Copy,PartialEq,Eq,uDebug,Persist)]
pub enum SerialError {
  /// The receive or transmit buffer overflowed
  BufferOverflow,

  /// We received something other than a stop bit when we were expecting one
  FrameError,

  /// The received data was framed correctly, but the calculated parity is
  /// wrong.
  ParityError,

  /// In automatic baud-rate detection, we cannot determine an accurate rate
  AutoBaudDetectFail,

  /// A Break was detected
  Break
}

/**
 * Response from the read handler callback.  A read handler can either return
 * a byte to be added to the input buffer, or it can return a Discard response
 * that will cause this byte to be discarded.
 */
#[derive(Copy,Clone,Debug)]
pub enum ReadHandlerResult<T> {
  /// Add the given byte to the serial receive buffer
  Buffer(T),
  /// Discard the byte
  Discard
}

/**
 * Callback called when a byte has been read
 */
pub type SerialReadEventHandlerFunction = fn(SerialPortIdentity, u8, Option<*const dyn Any>) ->ReadHandlerResult<u8>;

pub type SerialReadEventCallback = IsrCallback<SerialReadEventHandlerFunction,ReadHandlerResult<u8>>;

/**
 * Callback called when there is a serial error
 */
pub type SerialErrorEventHandlerFunction = fn(SerialPortIdentity, SerialError, Option<*const dyn Any>) ->();

pub type SerialErrorEventCallback = IsrCallback<SerialErrorEventHandlerFunction,()>;

/**
 * Callback called when a byte has been written
 */
pub type SerialWriteEventHandlerFunction = fn(SerialPortIdentity, Option<*const dyn Any>) ->();

pub type SerialWriteEventCallback = IsrCallback<SerialWriteEventHandlerFunction,()>;

pub trait SerialRxTx {
  /// Configure the serial port with the given mode
  fn mode(&mut self, mode:SerialPortMode) -> &mut Self;

  /// Enable or disable the receiver & transmitter
  fn enable_rxtx(&mut self, enable: bool) -> &mut Self;

  /// Set the callback handler for write notifications
  fn set_write_complete_callback(&self, handler: SerialWriteEventCallback);

  /// Set the callback handler for error
  fn set_error_callback(&self, handler: SerialErrorEventCallback);

  /// Set the callback handler for read notifications
  fn set_read_callback(&self, handler: SerialReadEventCallback);

  // Write a byte to the serial port.  May block if the transmit buffer is full.
  fn write_u8(&mut self, byte: u8);

  // Flush the transmit buffer (i.e. block until it is empty)
  fn flush(&mut self);

  // Read a byte from the serial port, blocking until a byte is available
  // if necessary.
  fn read_u8(&mut self) -> u8;

  /**
   * Read a byte from the serial port.  If no byte is available, return None
   * instead.
   */
  fn try_read_u8(&mut self) -> Option<u8>;
}

/**
 * Transmission in a no-interrupt context, i.e. write bytes even though
 * everything else is broken.  Used in internal contexts like the panic
 * handler.
 */
pub(crate) trait SerialNoInterruptTx {
  /**
   * Write a single byte synchronously, waiting until the hardware tells us it
   * is completed.
   */
  fn blocking_write_u8(&mut self, byte: u8);

  /**
   * Write a slice to the serial port.
   */
  fn blocking_write_slice(&mut self, bytes: &[u8]){
    for byte in bytes {
      self.blocking_write_u8(*byte);
    }
  }
}

// Code ======================================================================
#[allow(dead_code)]
impl BaudRate {
  /**
   * Convert a baud rate into a number of bits-per-second
   */
  pub fn to_bps(&self) -> u32 {
    match self {
      BaudRate::Baud300 => 300,
      BaudRate::Baud600 => 600,
      BaudRate::Baud1200 => 1200,
      BaudRate::Baud2400 => 2400,
      BaudRate::Baud4800 => 4800,
      BaudRate::Baud9600 => 9600,
      BaudRate::Baud14400 => 14400,
      BaudRate::Baud19200 => 19200,
      BaudRate::Baud28800 => 28800,
      BaudRate::Baud38400 => 38400,
      BaudRate::Baud57600 => 57600,
      BaudRate::Baud76800 => 76800,
      BaudRate::Baud115200 => 115200,
      BaudRate::Auto => 1
    }
  }

  #[cfg(not(feature="32BitDiv"))]
  pub(crate) fn baud_clock_table_index(&self) -> usize {
    match self {
      BaudRate::Baud300 => 1,
      BaudRate::Baud600 => 2,
      BaudRate::Baud1200 => 3,
      BaudRate::Baud2400 => 4,
      BaudRate::Baud4800 => 5,
      BaudRate::Baud9600 => 6,
      BaudRate::Baud14400 => 7,
      BaudRate::Baud19200 => 8,
      BaudRate::Baud28800 => 9,
      BaudRate::Baud38400 => 10,
      BaudRate::Baud57600 => 11,
      BaudRate::Baud76800 => 12,
      BaudRate::Baud115200 => 13,
      BaudRate::Auto => 0

    }
  }
}

#[cfg(target_arch="avr")]
pub mod base {
  pub mod usart {
    use core::any::Any;
    use core::cell::Cell;
    use avr_oxide::{v_read, v_write};
    use avr_oxide::hal::generic::serial::{SerialPortMode, SerialPortIdentity, SynchronousMode, BaudRate, Parity, StopBits, DataBits, SerialRxTx, ReadHandlerResult, SerialNoInterruptTx, SerialReadEventCallback, SerialWriteEventCallback, SerialErrorEventCallback};
    use avr_oxide::deviceconsts::clock::{MASTER_CLOCK_HZ, MASTER_CLOCK_PRESCALER};
    use core::marker::PhantomData;
    use avr_oxide::private::ringq::RingQ;

    /**
     * The AVR USART control register block
     */
    #[repr(C)]
    pub struct AtmelUsartControl {
      pub(crate) rxdatal: u8,
      pub(crate) rxdatah: u8,
      pub(crate) txdatal: u8,
      pub(crate) txdatah: u8,
      pub(crate) status: u8,
      pub(crate) ctrla: u8,
      pub(crate) ctrlb: u8,
      pub(crate) ctrlc: u8,
      pub(crate) baud: u16,
      pub(crate) ctrld: u8,
      pub(crate) dbgctrl: u8,
      pub(crate) evctrl: u8,
      pub(crate) txplctrl: u8,
      pub(crate) rxplctrl: u8
    }

    /**
     * Trait implemented to allow us to set portmux control bits
     */
    pub trait MuxControl {
      /**
       * Set the relevant hardware controls to route this appropriately.
       */
      fn route(&self);
    }

    pub struct AtmelUsart<M,const BUF_SIZE: usize>
    where
      M: MuxControl
    {
      pub(crate) control: &'static mut AtmelUsartControl,
      pub(crate) read_handler: Cell<SerialReadEventCallback>,
      pub(crate) write_complete_handler: Cell<SerialWriteEventCallback>,
      pub(crate) error_handler: Cell<SerialErrorEventCallback>,
      pub(crate) phantom: PhantomData<M>,

      pub(crate) rx_buf: RingQ<u8,BUF_SIZE>,
      pub(crate) tx_buf: RingQ<u8,BUF_SIZE>
    }

    /**
     * A no-op function that is the default handler for the serial
     * read interrupt callbacks.
     */
    #[inline(never)]
    pub(crate) fn read_nop_handler(_src: SerialPortIdentity, byte: u8, _udata: Option<*const dyn Any>) -> ReadHandlerResult<u8> {
      ReadHandlerResult::Buffer(byte)
    }

    impl<M,const BUF_SIZE:usize> AtmelUsart<M,BUF_SIZE>
    where
      M: MuxControl
    {
      pub fn mux(&mut self, mux: M) -> &mut Self {
        mux.route();

        self
      }

      /**
       * Enable or disable the Data Register Empty interrupt.  This will
       * send interrupts continuously as long as there is no data to send,
       * so we need to turn if off when we have nothing left and turn it on
       * where there might be more.  try_send_u8_from_buffer more or less
       * handles this for us.
       */
      #[inline(always)]
      pub(crate) unsafe fn dre_int_enable(&mut self, enabled: bool) {
        let ctrla = v_read!(u8, self.control.ctrla);

        match enabled {
          true  => v_write!(u8, self.control.ctrla, ctrla | 0b00100000),
          false => v_write!(u8, self.control.ctrla, ctrla & 0b11011111),
        };
      }

      /**
       * Try to send a byte from our buffer.  It will only do so if our USART's
       * transmit buffer is free, and also only if we have something to send :).
       *
       * Returns true if we succesfully dispatched a byte, false otherwise.
       */
      #[inline(always)]
      pub(crate) unsafe fn isr_try_send_u8_from_buffer(&mut self) -> bool {
        let byte = self.tx_buf.consume_nolock();

        match byte {
          Some(byte) => {
            v_write!(u8, self.control.txdatah, 0x00);
            v_write!(u8, self.control.txdatal, byte);
            true
          },
          None => {
            // Nothing to send, so disable the DRE interrupt

            false
          }
        }
      }

      /**
       * To to receive a byte into our buffer.  Returns true if we succeeded,
       * false if there was a buffer overflow.
       */
      #[inline(always)]
      pub(crate) unsafe fn isr_try_receive_u8_to_buffer(&mut self, byte: u8) -> bool {
        self.rx_buf.append_nolock(byte).is_ok()
      }
    }

    /**
     * Lookup table of asynchronous baud rate values for the USART, required
     * if 32bit maths is not working (as at the time of writing for Rust on AVR)
     */
    #[cfg(not(feature="32BitDiv"))]
    const SER_ASYNCH_CLOCK_RATES: [u16; 14] = [
      0,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *    300u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *    600u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *   1200u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *   2400u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *   4800u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *   9600u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *  14400u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *  19200u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *  28800u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *  38400u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *  57600u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 *  76800u32)) + 1) as u16,
      ((((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32) * 64u32) / (16u32 * 115200u32)) + 1) as u16,
    ];

    /**
     * Lookup table of synchronous baud rate values for the USART, required
     * if 32bit maths is not working (as at the time of writing for Rust on AVR)
     */
    #[cfg(not(feature="32BitDiv"))]
    const SER_SYNCH_CLOCK_RATES: [u16; 14] = [
      0,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *    300u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *    600u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *   1200u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *   2400u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *   4800u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *   9600u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *  14400u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *  19200u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *  28800u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *  38400u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *  57600u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 *  76800u32)) + 1) as u16,
      (((MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32)  / (2u32 * 115200u32)) + 1) as u16,
    ];


    impl SerialPortMode {
      /**
       * Return the value to write to the USART's `baud` register to configure
       * this speed.
       */
      #[cfg(feature="32BitDiv")]
      pub fn avr_baud_register_value(&self) -> u16 {
        // Some would have us do these calcs using floating point arithmetic.
        // We do not for three reasons -
        // 1. It involves linking in a raft of floating point library code,
        // 2. Floating point maths is irretrievably broken with LLVM-avr at
        //    the moment.
        // 3. It's entirely unnecessary


        let integer_clk_per = MASTER_CLOCK_HZ / MASTER_CLOCK_PRESCALER as u32;

        match self {
          SerialPortMode::Asynch(baud, _, _, _) => {
            let integer_baud_clock = (integer_clk_per * 64u32) / (16u32 * baud.to_bps() as u32)+1;
            integer_baud_clock as u16
          },
          SerialPortMode::Synch(mode) => {
              match mode {
              SynchronousMode::Master(baud) => {
                let integer_baud_clock = (integer_clk_per / (2u32 * baud.to_bps() as u32)) + 1;

                (integer_baud_clock as u16) << 6
              }
              SynchronousMode::Slave => {
                0
              }
            }
          }
        }
      }

      #[cfg(not(feature="32BitDiv"))]
      pub fn avr_baud_register_value(&self) -> u16 {
        // *Note*: 32-bit integer divide is broken on the AVR.  Honestly (rolleyes).
        // Constant expressions should be OK, being done at compile time, but
        // we can't allow an integer divide to make it into the code.  So we
        // need to use a lookup table
        match self {
          SerialPortMode::Asynch(baud, _, _, _) => {
            SER_ASYNCH_CLOCK_RATES[baud.baud_clock_table_index()]
          },
          SerialPortMode::Synch(mode) => {
            match mode {
              SynchronousMode::Master(baud) => {
                SER_SYNCH_CLOCK_RATES[baud.baud_clock_table_index()] << 6
              }
              SynchronousMode::Slave => {
                0
              }
            }
          }
        }
      }

      /**
       * Return the value to write to the `rxmode` bits of the USART's `ctrlb`
       * register to configure this mode.
       */
      pub fn avr_ctrlb_rxmode(&self) -> u8 {
        match self {
          SerialPortMode::Asynch(baud, _, _, _) => {
            match baud {
              BaudRate::Auto => 0x02 << 1,
              _ => 0x00
            }
          },
          SerialPortMode::Synch(_) => 0x00
        }
      }

      /**
       * Return the value to write the `ctrlc` register of the USART to
       * configure this mode
       */
      #[allow(clippy::unusual_byte_groupings)]
      pub fn avr_ctrlc(&self) -> u8 {
        match self {
          SerialPortMode::Asynch(_baud, dbits, parity, sbits) => {
            let mut ctrlc: u8 = 0b00_000000; // Asynchronous USART mode
            ctrlc |= match parity {
              Parity::None => 0b00_00_0000,
              Parity::Even => 0b00_10_0000,
              Parity::Odd =>  0b00_11_0000
            };
            ctrlc |= match sbits {
              StopBits::Bits1 => 0b0000_0_000,
              StopBits::Bits2 => 0b0000_1_000
            };
            ctrlc |= match dbits {
              DataBits::Bits5 => 0x00,
              DataBits::Bits6 => 0x01,
              DataBits::Bits7 => 0x02,
              DataBits::Bits8 => 0x03,
              DataBits::Bits9LE => 0x06,
              DataBits::Bits9HE => 0x07
            };
            ctrlc
          },
          SerialPortMode::Synch(_) => {
            let ctrlc: u8 = 0b01_000000; // Synchronous USART mode
            ctrlc
          }
        }
      }
    }
    impl<M,const BUF_SIZE:usize> SerialRxTx for AtmelUsart<M,BUF_SIZE>
    where
      M: MuxControl
    {
      fn mode(&mut self, mode: SerialPortMode) -> &mut Self {
        unsafe {
          // The datasheet says all USART configuration should be done with
          // global interrupts disabled, so...
          avr_oxide::hal::concurrency::interrupt::isolated(||{
            // Disable all interrupts
            v_write!(u8, self.control.ctrla, 0x00);

            // Wait until any pending writes are complete
            while (v_read!(u8, self.control.status) & 0b00100000) == 0 {
              #![allow(deprecated)] // until asm! is implemented for AVR
              llvm_asm!("nop" :::: "volatile" )
            }

            // Disable receiver/transmitter
            v_write!(u8, self.control.ctrlb, 0x00);

            // Clear the read buffer
            let _discard = v_read!(u8, self.control.rxdatah);
            let _discard = v_read!(u8, self.control.rxdatal);

            // OK, now we are clean and ready to configure

            // Set the baud rate register
            v_write!(u16, self.control.baud, mode.avr_baud_register_value());

            // Set the data comms mode
            v_write!(u8, self.control.ctrlc, mode.avr_ctrlc());

            // Set (harcode) auto-baud window size to the most permissive
            v_write!(u8, self.control.ctrld, 0b11000000);

            // Enable receive & transmit interrupts.
            //
            // Data Register Empty is disabled, we'll enable it when we actually
            // send something.
            v_write!(u8, self.control.ctrla, 0b11000000);
          });
        }
        self
      }

      /**
       * Enable (or disable) receiver/transmitter.  Call this after you have set
       * up the mode, but also after you have configured the receive/transmit
       * pins to input/output as required.
       */
      fn enable_rxtx(&mut self, enable: bool) -> &mut Self {
        unsafe {
          let ctrlb = v_read!(u8, self.control.ctrlb);

          match enable {
            true  => v_write!(u8, self.control.ctrlb, 0b11000000 | ctrlb),
            false => v_write!(u8, self.control.ctrlb, 0b00111111 & ctrlb),
          }
        }
        self
      }

      fn set_write_complete_callback(&self, callback: SerialWriteEventCallback) {
        avr_oxide::hal::concurrency::interrupt::isolated(||{
          self.write_complete_handler.replace(callback);
        });
      }

      fn set_read_callback(&self, callback: SerialReadEventCallback) {
        avr_oxide::hal::concurrency::interrupt::isolated(||{
          self.read_handler.replace(callback);
        });
      }

      fn set_error_callback(&self, callback: SerialErrorEventCallback) {
        avr_oxide::hal::concurrency::interrupt::isolated(||{
          self.error_handler.replace(callback);
        });
      }


      /**
       * Attempt to write a byte.  May block if the tx buffer is full.
       */
      fn write_u8(&mut self, byte: u8) {
        match self.tx_buf.append_nolock(byte) {
          Ok(()) => {
            // Byte added; nothing more for us to do!
          }
          Err(_e) => {
            // Hmm, byte couldn't be added.  Maybe our buffer is full - let's
            // try flushing and then block until we can add it.
            self.flush();
            self.tx_buf.append_blocking_nolock(byte);
          }
        }
      }

      /**
       * Flush the buffer (that is, start the interrupt handler which will
       * empty the buffer.)
       */
      fn flush(&mut self) {
        unsafe {
          // Enable the Data Register Empty interrupt, which will consume
          // any bytes in the buffer.
          self.dre_int_enable(true);
        }
      }

      /**
       * Read a byte from the input buffer, blocking until a byte is
       * available if the buffer is empty.
       */
      fn read_u8(&mut self) -> u8 {
        self.rx_buf.consume_blocking_nolock()
      }

      /**
       * Try to read a byte from the input buffer, returns Some(byte) if
       * there is a byte available, or None if not.
       */
      fn try_read_u8(&mut self) -> Option<u8> {
        self.rx_buf.consume_nolock()
      }
    }

    impl<M,const BUF_SIZE:usize> SerialNoInterruptTx for AtmelUsart<M,BUF_SIZE>
    where
      M: MuxControl
    {
      fn blocking_write_u8(&mut self, byte: u8) {
        unsafe {
          // Wait until the data register is ready to accept a byte
          while (v_read!(u8, self.control.status) & 0b00100000) == 0 {
            #![allow(deprecated)] // until asm! is implemented for AVR
            llvm_asm!("nop" :::: "volatile" )
          }
          // Now write the byte
          v_write!(u8, self.control.txdatah, 0);
          v_write!(u8, self.control.txdatal, byte);
        }
      }
    }

    #[doc(hidden)]
    #[macro_export]
    macro_rules! atmel_usart_tpl {
      ($usartref:expr, $srcref:expr, $muxtype:ty, $rxtxbufsize:expr, $isr_rx:ident, $isr_tx:ident, $isr_dre:ident) => {
        use avr_oxide::hal::generic::serial::{SerialError, ReadHandlerResult,SerialReadEventCallback,SerialWriteEventCallback,SerialErrorEventCallback};
        use avr_oxide::hal::generic::serial::base::usart::{ AtmelUsart, read_nop_handler };
        use avr_oxide::{mut_singleton,v_read,isr_cb_invoke};
        use avr_oxide::private::ringq::RingQ;
        use core::marker::PhantomData;
        use core::cell::Cell;

        pub type SerialImpl = AtmelUsart<$muxtype,$rxtxbufsize>;

        mut_singleton!(
          AtmelUsart<$muxtype,$rxtxbufsize>,
          INSTANCE,
          instance,
          AtmelUsart {
            control: core::mem::transmute($usartref),
            read_handler: Cell::new(SerialReadEventCallback::Function(read_nop_handler)),
            write_complete_handler: Cell::new(SerialWriteEventCallback::Nop(())),
            error_handler: Cell::new(SerialErrorEventCallback::Nop(())),
            phantom: PhantomData::default(),
            rx_buf: RingQ::new(),
            tx_buf: RingQ::new()
          });

        /**
         * Receive interrupt handler. Checks the type of interrupt, if it's
         * an error type we'll call the read handler with a suitable Err
         * instance.  If it's receive complete, we will read the byte and
         * send that to the callback.  In all cases, you need to have
         * set a `set_read_callback` of course.
         */
        #[no_mangle]
        pub unsafe extern "avr-interrupt" fn $isr_rx() {
          avr_oxide::hal::concurrency::interrupt::isr(||{
            match &mut INSTANCE {
              None => {},
              Some(atmelusart) => {
                let trigger = v_read!(u8, atmelusart.control.status);

                if (trigger & 0b10000000) > 0 { // Receive complete
                  // We MUST read the byte to clear the interrupt
                  let byte = v_read!(u8, atmelusart.control.rxdatal);
                  // Pass it to the user handler, and then buffer it if needed
                  match isr_cb_invoke!(atmelusart.read_handler.get(), $srcref, byte) {
                    ReadHandlerResult::Buffer(byte) => {
                      if !atmelusart.isr_try_receive_u8_to_buffer(byte) {
                        isr_cb_invoke!(atmelusart.error_handler.get(), $srcref, SerialError::BufferOverflow);
                      }
                    },
                    ReadHandlerResult::Discard => {}
                  }
                }
                if (trigger & 0b00001000) > 0 { // Baud sync fail
                  isr_cb_invoke!(atmelusart.error_handler.get(), $srcref, SerialError::AutoBaudDetectFail);
                }
                if (trigger & 0b00000010) > 0 { // Break detected
                  isr_cb_invoke!(atmelusart.error_handler.get(), $srcref, SerialError::Break);
                }

                // Clear all the other flags we don't care about
                //
                // @todo Note, in future we should care about them ;).
                core::ptr::write_volatile(&mut atmelusart.control.status as *mut u8,
                                          0b00011010);
              }
            }
          })
        }
        /**
         * Transmit interrupt handler.  If it's a transmit-complete interrupt,
         * we will call the write handler callback on this port (if one has
         * been set with `set_write_callback` method.)
         */
        #[no_mangle]
        pub unsafe extern "avr-interrupt" fn $isr_tx() {
          avr_oxide::hal::concurrency::interrupt::isr(||{
            // Note that this interrupt is cleared automatically on return from
            // the service routine.
          })
        }

        /**
         * Data Register Empty interrupt handler.  We will try to send a
         * byte from our transmit buffer, if that fails we will turn off
         * the interrupt.
         */
        #[no_mangle]
        pub unsafe extern "avr-interrupt" fn $isr_dre() {
          avr_oxide::hal::concurrency::interrupt::isr(||{
            match &mut INSTANCE {
              None => {},
              Some(atmelusart) => {
                match atmelusart.isr_try_send_u8_from_buffer() {
                  true => {
                    // Success, leave the interrupt enabled
                  },
                  false => {
                    // Nothing to send, so let's disable the interrupt
                    // and also send a notification
                    atmelusart.dre_int_enable(false);
                    isr_cb_invoke!(atmelusart.write_complete_handler.get(), $srcref);
                  }
                }
              }
            }
          })
        }
      }
    }
  }
}

// Tests =====================================================================
#[cfg(not(target_arch="avr"))]
pub mod base {
  pub mod usart {
    use avr_oxide::hal::generic::serial::{SerialPortMode, SerialPortIdentity, BaudRate, Parity, StopBits, DataBits, SerialRxTx, SerialWriteEventCallback, SerialReadEventCallback, SerialError, ReadHandlerResult, SerialErrorEventCallback};
    use core::marker::PhantomData;
    use std::fmt::{Display, Formatter};

    /**
     * The dummy USART control register block
     */
    #[repr(C)]
    pub struct DummyUsartControl {
    }

    /**
     * Trait implemented to allow us to set portmux control bits
     */
    pub trait MuxControl {
      /**
       * Set the relevant hardware controls to route this appropriately.
       */
      fn route(&self);
    }

    #[allow(dead_code)]
    pub struct DummyUsart<M,const BUF_SIZE: usize>
    where
      M: MuxControl
    {
      pub(crate) control: DummyUsartControl,
      pub(crate) phantom: Option<PhantomData<M>>
    }

    /**
     * A no-op function that is the default handler for the serial
     * read interrupt callbacks.
     */
    pub(crate) fn read_nop_handler(_src: SerialPortIdentity, byte: u8) -> ReadHandlerResult<u8> {
      ReadHandlerResult::Buffer(byte)
    }
    /**
     * A no-op function that is the default handler for the serial
     * write interrupt callbacks.
     */
    pub(crate) fn write_nop_handler(_src: SerialPortIdentity) -> () {
    }
    /**
     * A no-op function that is the default handler for the serial
     * error interrupt callbacks.
     */
    pub(crate) fn error_nop_handler(_src: SerialPortIdentity, _error: SerialError) -> () {
    }

    impl<M,const BUF_SIZE:usize> DummyUsart<M,BUF_SIZE>
    where
      M: MuxControl
    {
      pub fn mux(&mut self, mux: M) -> &mut Self {
        mux.route();

        self
      }
    }

    impl Display for SerialPortMode {
      fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
          SerialPortMode::Asynch(baud, dbits, parity, stopbits) => {
            f.write_str(&format!("{} {}{}{}", baud, dbits, parity, stopbits))
          }
          SerialPortMode::Synch(_) => {
            f.write_str("Synchronous")
          }
        }
      }
    }
    impl Display for BaudRate {
      fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        f.write_str(match self {
          BaudRate::Baud300 => "300",
          BaudRate::Baud600 => "600",
          BaudRate::Baud1200 => "1200",
          BaudRate::Baud2400 => "2400",
          BaudRate::Baud4800 => "4800",
          BaudRate::Baud9600 => "9600",
          BaudRate::Baud14400 => "14400",
          BaudRate::Baud19200 => "19200",
          BaudRate::Baud28800 => "28800",
          BaudRate::Baud38400 => "38400",
          BaudRate::Baud57600 => "57600",
          BaudRate::Baud76800 => "76800",
          BaudRate::Baud115200 => "115200",
          BaudRate::Auto => "Auto"
        })
      }
    }
    impl Display for DataBits {
      fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        f.write_str(match self {
          DataBits::Bits5 => "5",
          DataBits::Bits6 => "6",
          DataBits::Bits7 => "7",
          DataBits::Bits8 => "8",
          DataBits::Bits9LE => "9le",
          DataBits::Bits9HE => "9he"
        })
      }
    }
    impl Display for Parity {
      fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        f.write_str(match self {
          Parity::None => "N",
          Parity::Even => "E",
          Parity::Odd => "O"
        })
      }
    }
    impl Display for StopBits {
      fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        f.write_str(match self {
          StopBits::Bits1 => "1",
          StopBits::Bits2 => "2"
        })
      }
    }

    impl<M,const BUF_SIZE:usize> SerialRxTx for DummyUsart<M,BUF_SIZE>
      where
        M: MuxControl
    {
      fn mode(&mut self, mode: SerialPortMode) -> &mut Self {
        println!("*** USART: Set port mode to {}", mode);
        self
      }

      fn enable_rxtx(&mut self, enable: bool) -> &mut Self {
        println!("*** USART: Rx/Tx enable set to {}", enable);
        self
      }

      fn set_write_complete_callback(&self, callback: SerialWriteEventCallback) {
        println!("*** USART: Write callback set to {:?}", callback);
      }

      fn set_read_callback(&self, callback: SerialReadEventCallback) {
        println!("*** USART: Write callback set to {:?}", callback);
      }

      fn set_error_callback(&self, callback: SerialErrorEventCallback) {
        println!("*** USART: Write callback set to {:?}", callback);
      }


      /**
       * Attempt to write a byte.  May block if the tx buffer is full.
       */
      fn write_u8(&mut self, byte: u8) {
        println!("*** USART: Wrote '{}'", byte as char);
        eprint!("{}", byte as char);
      }

      fn flush(&mut self) {
        todo!()
      }

      fn read_u8(&mut self) -> u8 {
        todo!()
      }

      fn try_read_u8(&mut self) -> Option<u8> {
        todo!()
      }
    }

    #[doc(hidden)]
    #[macro_export]
    macro_rules! atmel_usart_tpl {
      ($usartref:expr, $srcref:expr, $muxtype:ty, $rxtxbufsize:expr, $isr_rx:ident, $isr_tx:ident, $isr_dre:ident) => {
        use avr_oxide::hal::generic::serial::base::usart::{ DummyUsart, DummyUsartControl, read_nop_handler, write_nop_handler, error_nop_handler };

        pub type SerialImpl = DummyUsart<$muxtype,$rxtxbufsize>;

        static mut INSTANCE : DummyUsart<$muxtype,$rxtxbufsize> = DummyUsart {
          control: DummyUsartControl {},
          phantom: None
        };

        pub fn instance() -> &'static mut SerialImpl {
          unsafe {
            &mut INSTANCE
          }
        }
      }
    }
  }
}