/* mod.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! ATmega4809 implementations of the generic device traits
#![allow(unused)]
use avr_oxide::hal::generic::debugled::DebugLed;
use avr_oxide::hal::generic::serial::SerialNoInterruptTx;

pub mod cpu;
pub mod timer;
pub mod port;
pub mod hardware;
pub mod debugled;
pub mod watchdog;
pub mod serial;
pub mod eeprom;


#[cfg(feature="panic_handler")]
#[panic_handler]
#[cfg(target_arch="avr")]
fn panic(info: &core::panic::PanicInfo) -> ! {
  #![allow(deprecated)] // until asm! is implemented for AVR
  unsafe {
    avr_oxide::hal::concurrency::interrupt::disable_interrupts();

    #[cfg(feature="panicout")]
    {
      use avr_oxide::util::{debug_print, debug_print_u16};
      // This doesn't look too pretty, but we can't use any of the standard
      // formatting routines because they require an allocator, which even
      // if we have one, is probably broken right now.
      // Plus the standard formatter routines don't know that we can't use
      // 32 bit arithmetic at the moment 😖
      debug_print("\n\nPanic:\n");
      if let Some(location) = info.location() {
        debug_print("   --> ");
        debug_print(&location.file());
        debug_print(":");
        debug_print_u16(location.line() as u16);
        debug_print(":");
        debug_print_u16(location.column() as u16);
        debug_print("\n");
      }
    }


    llvm_asm!("break" :::: "volatile");

    loop {
      #[cfg(feature="arduino")]
      avr_oxide::hal::atmega4809::debugled::PinE2DebugLed::toggle();

      for _i in 0..255 {
        avr_oxide::hal::generic::busy_loop(255);
      }
    }
  }
}

const ADDR_CPU:     u16 = 0x0030;
const ADDR_CLKCTRL: u16 = 0x0060;
const ADDR_PORTA:   u16 = 0x0400;
const ADDR_PORTB:   u16 = 0x0420;
const ADDR_PORTC:   u16 = 0x0440;
const ADDR_PORTD:   u16 = 0x0460;
const ADDR_PORTE:   u16 = 0x0480;
const ADDR_PORTF:   u16 = 0x04A0;
const ADDR_PORTMUX: u16 = 0x05E0;
const ADDR_USART0:  u16 = 0x0800;
const ADDR_USART1:  u16 = 0x0820;
const ADDR_USART2:  u16 = 0x0840;
const ADDR_USART3:  u16 = 0x0860;
const ADDR_TCB0:    u16 = 0x0A80;
const ADDR_TCB1:    u16 = 0x0A90;
const ADDR_TCB2:    u16 = 0x0AA0;
const ADDR_TCB3:    u16 = 0x0AB0;
const ADDR_WDT:     u16 = 0x0100;
const ADDR_RTC:     u16 = 0x0140;
const ADDR_NVMCTRL: u16 = 0x1000;
const ADDR_USERROW_PBUF: u16 = 0x1300;
const ADDR_EEPROM_PBUF:  u16 = 0x1400;
const EEPROM_PAGESIZE : usize = 64;
const EEPROM_PAGES    : usize = 4;
const USERROW_PAGESIZE: usize = 64;
const USERROW_PAGES   : usize = 1;