/* mod.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Simple IO traits (a very, very, very cutdown version of std::io).  In due
//! course hopefully somewhat /less/ cutdown, and maybe even derived from
//! std::io in the style of the `core_io` crate, but right now this is what
//! we've got.
//!
//! (We can't use `core_io` at the moment because the build scripts do not
//! recognise our custom toolchain; if we ever get to a place where the
//! regular Rust toolchain works on AVR again, we may be able to revisit this.)

// Imports ===================================================================


// Declarations ==============================================================
/// Errors that can result from our IO routines
pub enum IoError {
  /// We have reached the end of a file for reading
  EndOfFile,

  /// We have no remaining free space for writing
  NoFreeSpace,

  /// Generic IO error
  Unknown
}

pub type Result<T> = core::result::Result<T,IoError>;

pub trait Read {
  fn read(&mut self, _buf: &mut [u8]) -> crate::io::Result<usize>;
}

pub trait Write {
  fn write(&mut self, buf: &[u8]) -> crate::io::Result<usize>;
  fn flush(&mut self) -> crate::io::Result<()>;
}


// Code ======================================================================

#[cfg(not(target_arch="avr"))]
impl Read for std::io::Stdin {
  fn read(&mut self, buf: &mut [u8]) -> Result<usize> {
    match std::io::Read::read(self, buf) {
      Ok(b) => Ok(b),
      Err(_e) => Err(IoError::Unknown)
    }
  }
}
#[cfg(not(target_arch="avr"))]
impl Write for std::io::Stdout {
  fn write(&mut self, buf: &[u8]) -> Result<usize> {
    match std::io::Write::write(self, buf) {
      Ok(b) => Ok(b),
      Err(_e) => Err(IoError::Unknown)
    }
  }

  fn flush(&mut self) -> Result<()> {
    std::io::Write::flush(self);
    Ok(())
  }
}
#[cfg(not(target_arch="avr"))]
impl Write for std::io::Stderr {
  fn write(&mut self, buf: &[u8]) -> Result<usize> {
    match std::io::Write::write(self, buf) {
      Ok(b) => Ok(b),
      Err(_e) => Err(IoError::Unknown)
    }
  }

  fn flush(&mut self) -> Result<()> {
    std::io::Write::flush(self);
    Ok(())
  }
}

// Tests =====================================================================