/* mod.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Generic traits for the various devices that can be attached to our AVR
//! device
pub mod callback;
pub mod timer;
pub mod port;
pub mod debugled;
pub mod watchdog;
pub mod cpu;
pub mod serial;
pub mod eeprom;


#[allow(unused_assignments)]
#[cfg(target_arch="avr")]
pub(in crate) fn busy_loop(mut c: u16) {
  unsafe {
    #![allow(deprecated)] // until asm! is implemented for AVR
    llvm_asm!("1: sbiw $0,1\n\tbrne 1b"
                     : "=w"(c)
                     : "0"(c)
                     :
                     : "volatile"
                 );
  }
}

#[cfg(not(target_arch="avr"))]
#[allow(dead_code)]
pub(in crate) fn busy_loop(c: u16) {
  let mut count = 0u16;
  let volatile = &mut count as *mut u16;

  for _i in 0..c {
    unsafe {
      std::ptr::write_volatile(volatile, count+1);
    }
  }
}

#[lang = "eh_personality"]
#[no_mangle]
#[cfg(target_arch="avr")]
#[allow(clippy::missing_safety_doc)]
pub unsafe extern "C" fn rust_eh_personality() {
}

/**
 * Volatile write to a variable by reference
 */
#[macro_export]
macro_rules! v_write {
  ($type:ty, $ref:expr, $val:expr) => {
    core::ptr::write_volatile(&mut $ref as *mut $type, $val)
  }
}
/**
 * Volatile read from a variable by reference
 */
#[macro_export]
macro_rules! v_read {
  ($type:ty, $ref:expr) => {
    core::ptr::read_volatile(&$ref as *const $type)
  }
}
