/* hardware.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Provides a single point of access to the hardware devices associated with
//! the AVR CPU on which we are running.
//!
//! This struct is your first port of call for accessing a pin, timer, or
//! any other hardware component which we have support for.
//!
//! It is implemented as a global singleton, a reference is obtained using
//! the provided instance() function.  From here, you can then obtain
//! references to all the supported low-level hardware devices.
//!
//! # Example usage
//! ```rust
//! use avr_oxide::hal::atmega4809;
//!
//! fn use_hardware() {
//!   let hardware = atmega4809::hardware::instance();
//!
//!   // Get a reference to the EEPROM device
//!   let eeprom = &hardware.eeprom;
//!
//!   // Get a reference to Port D
//!   let portd = &hardware.portd;
//! }
//! ```

// Imports ===================================================================
use crate::hal::atmega4809;
use crate::mut_singleton;

// Declarations ==============================================================
pub struct Hardware {
  pub cpu: &'static mut atmega4809::cpu::CpuImpl,

  #[cfg(feature="tcb0")]
  pub timerb0: &'static mut atmega4809::timer::tcb0::TimerImpl,
  #[cfg(feature="tcb1")]
  pub timerb1: &'static mut atmega4809::timer::tcb1::TimerImpl,
  #[cfg(feature="tcb2")]
  pub timerb2: &'static mut atmega4809::timer::tcb2::TimerImpl,
  #[cfg(feature="tcb3")]
  pub timerb3: &'static mut atmega4809::timer::tcb3::TimerImpl,
  #[cfg(feature="rtc")]
  pub rtc: &'static mut atmega4809::timer::rtc::TimerImpl,

  pub porta: &'static atmega4809::port::porta::PortImpl,
  pub portb: &'static atmega4809::port::portb::PortImpl,
  pub portc: &'static atmega4809::port::portc::PortImpl,
  pub portd: &'static atmega4809::port::portd::PortImpl,
  pub porte: &'static atmega4809::port::porte::PortImpl,
  pub portf: &'static atmega4809::port::portf::PortImpl,

  pub watchdog: &'static mut atmega4809::watchdog::WatchdogImpl,

  pub eeprom: &'static mut atmega4809::eeprom::eeprom::EepromImpl,
  pub userrow: &'static mut atmega4809::eeprom::userrow::EepromImpl,

  #[cfg(feature="usart0")]
  pub usart0: &'static mut atmega4809::serial::usart0::SerialImpl,
  #[cfg(feature="usart1")]
  pub usart1: &'static mut atmega4809::serial::usart1::SerialImpl,
  #[cfg(feature="usart2")]
  pub usart2: &'static mut atmega4809::serial::usart2::SerialImpl,
  #[cfg(feature="usart3")]
  pub usart3: &'static mut atmega4809::serial::usart3::SerialImpl,
}

// Code ======================================================================
mut_singleton!(Hardware, HARDWARE, instance,
  Hardware {
    cpu: atmega4809::cpu::instance(),

    #[cfg(feature="tcb0")]
    timerb0: atmega4809::timer::tcb0::instance(),
    #[cfg(feature="tcb1")]
    timerb1: atmega4809::timer::tcb1::instance(),
    #[cfg(feature="tcb2")]
    timerb2: atmega4809::timer::tcb2::instance(),
    #[cfg(feature="tcb3")]
    timerb3: atmega4809::timer::tcb3::instance(),
    #[cfg(feature="rtc")]
    rtc: atmega4809::timer::rtc::instance(),

    eeprom: atmega4809::eeprom::eeprom::instance(),
    userrow: atmega4809::eeprom::userrow::instance(),

    porta: atmega4809::port::porta::instance(),
    portb: atmega4809::port::portb::instance(),
    portc: atmega4809::port::portc::instance(),
    portd: atmega4809::port::portd::instance(),
    porte: atmega4809::port::porte::instance(),
    portf: atmega4809::port::portf::instance(),

    watchdog: atmega4809::watchdog::instance(),

    #[cfg(feature="usart0")]
    usart0: atmega4809::serial::usart0::instance(),
    #[cfg(feature="usart1")]
    usart1: atmega4809::serial::usart1::instance(),
    #[cfg(feature="usart2")]
    usart2: atmega4809::serial::usart2::instance(),
    #[cfg(feature="usart3")]
    usart3: atmega4809::serial::usart3::instance(),
  }
);
