/* serialport.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Abstractions for serial I/O devices, implementing the Read/Write interfaces
//! provided by [`avr_oxide::io`].
//!
//! The SerialPort implementation supports the port mnultiplexing of the AVR
//! controller, meaning that an underlying USART may be connected to different
//! pins of the controller at runtime.  Thus the device must be instantiated
//! with both the USART device to use, and the pins which will be used for
//! receive and transmit.
//!
//! # Usage
//! ```no_run
//! # #![no_std]
//! # #![no_main]
//! #
//! # use avr_oxide::hal::atmega4809::hardware;
//! # use avr_oxide::alloc::boxed::Box;
//! # use avr_oxide::devices::UsesPin;
//! # use avr_oxide::devices::debouncer::Debouncer;
//! # use avr_oxide::devices::{ Handle, OxideLed, OxideButton, OxideSerialPort };
//! # use avr_oxide::hal::generic::serial::{BaudRate, DataBits, Parity, SerialPortMode, StopBits};
//! # use avr_oxide::io::Write;
//! # use avr_oxide::arduino;
//! #
//! # #[avr_oxide::main(chip="atmega4809")]
//! # pub fn main() {
//! #   let supervisor = avr_oxide::oxide::instance();
//! #   let hardware = hardware::instance();
//!   let arduino = arduino::nanoevery::Arduino::from(hardware);
//!
//!   // Configure the serial port early so we can get any panic!() messages
//!   let mut serial= OxideSerialPort::using_port_and_pins(arduino.usb_serial,
//!                                                        arduino.usb_serial_tx,
//!                                                        arduino.usb_serial_rx).mode(SerialPortMode::Asynch(BaudRate::Baud9600, DataBits::Bits8, Parity::None, StopBits::Bits1));
//!   serial.write(b"Welcome to AVRoxide\n");
//!
//! #    supervisor.run();
//! # }
//! ```


// Imports ===================================================================
use crate::hal::generic::serial::{SerialRxTx, ReadHandlerResult, SerialError, SerialPortMode, SerialPortIdentity, SerialWriteEventCallback, SerialReadEventCallback, SerialErrorEventCallback};
use crate::event::{EventSink, EventSource, OxideEvent, OxideEventEnvelope};
use core::marker::PhantomData;
use core::cell::RefCell;
use core::ops::DerefMut;
use ufmt::derive::uDebug;
use avr_oxide::devices::internal::StaticShareable;
use oxide_macros::Persist;
use crate::alloc::boxed::Box;
use crate::devices::Handle;
use crate::hal::generic::callback::IsrCallback;
use crate::hal::generic::port::{Pin, PinMode};
use crate::panic_if_none;
use crate::util::{BitField, BitIndex};

// Declarations ===============================================================
pub trait SerialCallback = FnMut(SerialPortIdentity, SerialState) -> ();

/**
 * State of the serial port sent with Oxide events.
 */
#[derive(PartialEq,Eq,Clone,Copy,uDebug,Persist)]
pub enum SerialState {
  /// There is data waiting in the buffer for someone to read it
  ReadAvailable,

  /// A serial error has been detected (framing, parity etc.)
  SerialCommsError,

  /// A *break* condition on the receive line has been detected
  BreakDetected,

  /// The receive buffer has overflowed (data was not consumed quickly
  /// enough...)
  ReadBufferOverflow
}

/**
 * Options for controlling how data is transmitted by the serial port device.
 */
pub type SerialOptions = BitField<1>;


/**
 * Encapsulation of a serial port exposing the standard io::Read/io::Write
 * traits.  Also generates Oxide event notifications.
 */
pub struct SerialPort<'sp,P,T,R,S>
where
  P: 'static + SerialRxTx,
  T: 'static + Pin,
  R: 'static + Pin,
  S: EventSink
{
  port: &'static mut P,
  tx: &'static T,
  rx: &'static R,
  phantom: PhantomData<S>,

  serial_options: SerialOptions,

  on_event: RefCell<Option<Box<dyn SerialCallback + 'sp>>>
}

impl<'sp,P,T,R,S> StaticShareable for SerialPort<'sp,P,T,R,S>
where
  P: 'static + SerialRxTx,
  T: 'static + Pin,
  R: 'static + Pin,
  S: EventSink
{}


// Code ======================================================================
const TX_FLUSH_ON_EOL: BitIndex =  BitIndex::bit(0);
const TX_CR_TO_CRLF:   BitIndex =  BitIndex::bit(1);
const TX_LF_TO_CRLF:   BitIndex =  BitIndex::bit(2);
const INTERACTIVE:     BitIndex =  BitIndex::bit(3);

/**
 * Configuration of various translation/usability options for the serial
 * port device.
 *
 * | Option         | Effect                                       |
 * | -------------- | -------------------------------------------- |
 * | `flush_on_eol` | Automatically flush when an end-of-line is transmitted |
 * | `cr_to_crlf`   | Translate sent CRs to CRLF |
 * | `lf_to_crlf`   | Translate sent LFs to CRLF |
 * | `interactive`  | This is an interactive terminal, reading from the device will automatically flush the transmit buffer first |
 */
impl SerialOptions {
  /// Indicate that the device should automatically flush when an end-of-line
  /// (CR or LF) character is sent
  /// is sent.
  pub fn set_flush_on_eol(&mut self, flag: bool) {
    self.set_or_clr(TX_FLUSH_ON_EOL, flag);
  }

  /// True iff the device should automatically flush when an LF character
  /// is sent.
  pub fn flush_on_eol(self) -> bool {
    self.is_set(TX_FLUSH_ON_EOL)
  }

  /// Indicate that CR should be translated to CRLF automatically
  pub fn set_cr_to_crlf(&mut self, flag: bool) {
    self.set_or_clr(TX_CR_TO_CRLF, flag);
  }

  /// True iff the device should automatically translate CR to CRLF.
  pub fn cr_to_crlf(&self) -> bool {
    self.is_set(TX_CR_TO_CRLF)
  }

  /// Indicate that LF should be translated to CRLF automatically
  pub fn set_lf_to_crlf(&mut self, flag: bool) {
    self.set_or_clr(TX_LF_TO_CRLF, flag);
  }

  /// True iff the device should automatically translate LF to CRLF.
  pub fn lf_to_crlf(&self) -> bool {
    self.is_set(TX_LF_TO_CRLF)
  }
  /// Indicate that this is an interactive terminal.  Effects:
  ///   * A call to read() will automatically flush the transmit buffer
  pub fn set_interactive(&mut self, flag: bool) {
    self.set_or_clr(INTERACTIVE, flag);
  }

  /// True iff this is an interative terminal
  pub fn interactive(&self) -> bool {
    self.is_set(INTERACTIVE)
  }

  /// Clear all settings that result in the data being translated in any
  /// way (e.g. CR to CRLF type translations.)  Use this to ensure a serial
  /// port is 'clean' for sending/receiving binary data.
  pub fn disable_translations(&mut self) {
    self.set_cr_to_crlf(false);
    self.set_lf_to_crlf(false);
  }
}

impl Default for SerialOptions {
  /**
   * Default transmit options:
   *  * DO flush automatically on End-of-Line
   *  * DO NOT translate CR to CRLF
   *  * DO NOT translate LF to CRLF
   *  * Interactive mode NOT enabled
   */
  fn default() -> Self {
    BitField::with_bits_set(&[TX_FLUSH_ON_EOL])
  }
}

impl<P,T,R,S> SerialPort<'_,P,T,R,S>
where
  P: 'static + SerialRxTx,
  T: 'static + Pin,
  R: 'static + Pin,
  S: EventSink
{
  /**
   * Create an instance of SerialPort that will communicate over the given
   * port and pins.
   */
  pub fn using_port_and_pins(port: &'static mut P, tx: &'static mut T, rx: &'static mut R) -> Self {
    rx.listen(IsrCallback::Nop(()));



    port.enable_rxtx(true);
    tx.set_mode(PinMode::Output);
    rx.set_mode(PinMode::InputFloating);

    SerialPort {
      port, tx, rx,
      phantom: PhantomData::default(),

      serial_options: SerialOptions::default(),

      on_event: RefCell::new(None)
    }
  }

  /**
   * Set the serial port comms parameters.
   */
  pub fn mode(self, mode: SerialPortMode) -> Self {
    // Disconnect our pins while we reconfigure mode to avoid splurging junk
    // out onto the line
    self.tx.set_mode(PinMode::InputFloating);
    self.rx.set_mode(PinMode::InputFloating);

    self.port.mode(mode);
    self.port.enable_rxtx(true);
    // Now we can enable the transmit pin again
    self.tx.set_mode(PinMode::Output);

    self
  }

  /**
   * Get the currently configured transmission options.
   */
  pub fn get_serial_options(&self) -> SerialOptions {
    self.serial_options
  }

  /**
   * Set the transmission options.
   */
  pub fn set_serial_options(&mut self, options: SerialOptions) {
    self.serial_options = options;
  }

  /**
   * Mutably access the serial options
   */
  pub fn serial_options(&mut self) -> &mut SerialOptions {
    &mut self.serial_options
  }
}

impl<P,T,R,S> EventSource for SerialPort<'_,P,T,R,S>
where
  P: 'static + SerialRxTx,
  T: 'static + Pin,
  R: 'static + Pin,
  S: EventSink
{
  fn listen(&'static self) {
    // We won't send any events on write.
    self.port.set_write_complete_callback(SerialWriteEventCallback::Nop(()));

    // When the serial port receives a byte, send a ReadAvailable message
    self.port.set_read_callback(SerialReadEventCallback::WithData(|src,byte, udata|{
      S::event(OxideEventEnvelope::to(unsafe {&*(panic_if_none!(udata) as *const SerialPort<P,T,R,S> as *const dyn EventSource)},
                                      OxideEvent::SerialEvent(src, SerialState::ReadAvailable)));
      ReadHandlerResult::Buffer(byte)
    },self as *const dyn core::any::Any));

    // When the serial port generates an error, send a suitable message
    self.port.set_error_callback(SerialErrorEventCallback::WithData(|src,error, udata|{
      S::event(OxideEventEnvelope::to(unsafe {&*(panic_if_none!(udata) as *const SerialPort<P,T,R,S> as *const dyn EventSource)},
      OxideEvent::SerialEvent(src, match error {
        SerialError::BufferOverflow => SerialState::ReadBufferOverflow,
        SerialError::Break => SerialState::BreakDetected,
        _ => SerialState::SerialCommsError
      })));
    }, self as *const dyn core::any::Any));
  }

  fn process_event(&self, evt: OxideEvent) {
    match (self.on_event.borrow_mut().deref_mut(), evt) {
      (Some(f), OxideEvent::SerialEvent(source, state)) => {
        (*f)(source,state)
      },
      _ => {}
    }
  }
}

impl<P,T,R,S> crate::io::Read for SerialPort<'_,P,T,R,S>
where
  P: 'static + SerialRxTx,
  T: 'static + Pin,
  R: 'static + Pin,
  S: EventSink
{
  fn read(&mut self, buf: &mut [u8]) -> crate::io::Result<usize> {
    let mut cnt = 0usize;

    if self.serial_options.interactive() {
      self.port.flush();
    }

    while cnt < buf.len() {
      match self.port.try_read_u8() {
        Some(c) => {
          buf[cnt] = c;
          cnt += 1;
        },
        None => {
          break;
        }
      }
    }

    Ok(cnt)
  }
}

impl<P,T,R,S> crate::io::Write for SerialPort<'_,P,T,R,S>
where
  P: 'static + SerialRxTx,
  T: 'static + Pin,
  R: 'static + Pin,
  S: EventSink
{
  fn write(&mut self, buf: &[u8]) -> crate::io::Result<usize> {
    let cnt : usize = 0;

    for byte in buf {

      // Special handling before we send the character
      match byte {
        b'\n' => {
          if self.serial_options.lf_to_crlf() {
            self.port.write_u8(b'\r');
          }
        },
        _ => {}
      }

      // Send the character
      self.port.write_u8(*byte);

      // Special handling after we send the character
      match byte {
        b'\r' => {
          if self.serial_options.cr_to_crlf() {
            self.port.write_u8(b'\n');
          }
          if self.serial_options.flush_on_eol() {
            self.port.flush();
          }
        },
        b'\n' => {
          if self.serial_options.flush_on_eol() {
            self.port.flush();
          }
        },
        _ => {}
      }
    }
    Ok(cnt)
  }

  fn flush(&mut self) -> crate::io::Result<()> {
    self.port.flush();
    Ok(())
  }
}

impl<P,T,R,S> ufmt_write::uWrite for SerialPort<'_,P,T,R,S>
  where
    P: 'static + SerialRxTx,
    T: 'static + Pin,
    R: 'static + Pin,
    S: EventSink
{
  type Error = core::convert::Infallible;

  fn write_str(&mut self, s: &str) -> Result<(), Self::Error> {
    let _ignore = crate::io::Write::write(self, s.as_bytes());
    Ok(())
  }
}

impl<P,T,R,S> crate::io::Write for Handle<SerialPort<'_,P,T,R,S>>
where
  P: 'static + SerialRxTx,
  T: 'static + Pin,
  R: 'static + Pin,
  S: EventSink
{
  fn write(&mut self, buf: &[u8]) -> crate::io::Result<usize> {
    Handle::deref_mut(self).write(buf)
  }

  fn flush(&mut self) -> crate::io::Result<()> {
    Handle::deref_mut(self).flush()
  }
}

impl<P,T,R,S> crate::io::Read for Handle<SerialPort<'_,P,T,R,S>>
where
  P: 'static + SerialRxTx,
  T: 'static + Pin,
  R: 'static + Pin,
  S: EventSink
{
  fn read(&mut self, buf: &mut [u8]) -> crate::io::Result<usize> {
    Handle::deref_mut(self).read(buf)
  }
}

impl<P,T,R,S> ufmt_write::uWrite for Handle<SerialPort<'_,P,T,R,S>>
  where
    P: 'static + SerialRxTx,
    T: 'static + Pin,
    R: 'static + Pin,
    S: EventSink
{
  type Error = core::convert::Infallible;

  fn write_str(&mut self, s: &str) -> Result<(), Self::Error> {
    Handle::deref_mut(self).write_str(s)
  }
}

