/* button.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Simple abstraction of a button attached to a GPIO port.
//!
//! # Usage
//! Create the button using one of the constructors provided by the [`avr_oxide::devices::UsesPin`]
//! trait.
//!
//! The AVR's internal pullup can be enabled/disabled using the
//! `pullup` method.
//!
//! State can be queried using the `is_pressed` or `state` methods, or
//! alternatively a closure can be provided as a callback to be called
//! whenever the button's state changes using the `on_click` method.
//!
//! ```rust,no_run
//! # #![no_std]
//! # #![no_main]
//! #
//! # use avr_oxide::hal::atmega4809::hardware;
//! # use avr_oxide::alloc::boxed::Box;
//! # use avr_oxide::devices::UsesPin;
//! # use avr_oxide::devices::debouncer::Debouncer;
//! # use avr_oxide::devices::{ Handle, OxideButton, button::ButtonState };
//! # use avr_oxide::arduino;
//!
//! #[avr_oxide::main(chip="atmega4809")]
//! pub fn main() {
//!   let supervisor = avr_oxide::oxide::instance();
//!   let hardware = hardware::instance();
//!   let arduino = arduino::nanoevery::Arduino::from(hardware);
//!
//!   let mut green_button = Handle::new(OxideButton::using(Debouncer::with_pin(arduino.a2)));
//!
//!   green_button.on_click(Box::new(move |_pinid, state|{
//!     match state {
//!       ButtonState::Released => {
//!         // Do something
//!       },
//!       ButtonState::Pressed => {
//!         // Do something
//!       },
//!       ButtonState::Unknown => {
//!         // Probably don't do anything
//!       },
//!     }
//!   }));
//!
//!   // Tell the supervisor which devices to listen to
//!   supervisor.listen_handle(green_button);
//!
//!   // Now enter the event loop
//!   supervisor.run();
//! }
//! ```


// Imports ===================================================================
use crate::hal::generic::port::{Pin, PinMode, InterruptMode, PinIdentity};
use crate::event::{EventSink, EventSource, OxideEvent, OxideEventEnvelope};
use core::marker::PhantomData;
use core::ops::DerefMut;
use core::cell::RefCell;
use ufmt::derive::uDebug;
use avr_oxide::devices::internal::StaticShareable;
use oxide_macros::Persist;
use crate::hal::generic::callback::IsrCallback;
#[cfg(feature="alloc_ftr")]
use crate::alloc::boxed::Box;
use crate::devices::UsesPin;
use crate::panic_if_none;
use crate::util::OwnOrBorrowMut;

// Declarations ==============================================================

/**
 * Button states passed in Oxide events.
 */
#[derive(PartialEq,Eq,Clone,Copy,uDebug,Persist)]
pub enum ButtonState {
  /// Button was pressed
  Pressed,
  /// Button was released
  Released,
  /// Button is in an unknown state
  Unknown
}

/**
 * Encapsulation of a Button attached to a standard GPIO pin.  Generates Oxide
 * events on button events (press/release.)
 */
pub struct Button<'b, P, S>
where
  P: 'static + Pin,
  S: EventSink
{
  pin: OwnOrBorrowMut<'static,P>,
  phantom: PhantomData<S>,

  on_click: RefCell<Option<Box<dyn FnMut(PinIdentity,ButtonState)->() + 'b>>>
}

impl<'b,P,S> StaticShareable for Button<'b, P, S>
  where
    P: 'static + Pin,
    S: EventSink
{}

// Code ======================================================================
impl<'b,P,S> UsesPin<P> for Button<'b,P,S>
where
  P: Pin,
  S: 'static + EventSink
{
  fn using<OP: Into<OwnOrBorrowMut<'static, P>>>(pin: OP) -> Self {
    let pin : OwnOrBorrowMut<P> = pin.into();

    pin.set_mode(PinMode::InputFloating);
    pin.set_interrupt_mode(InterruptMode::BothEdges);
    Button {
      pin,
      phantom: PhantomData::default(),
      on_click: RefCell::new(None)
    }
  }
}

impl<'b,P,S> Button<'b,P,S>
where
  P: Pin,
  S: 'static + EventSink
{
  /**
   * Have the Oxide supervisor execute the given closure every time an event
   * is generated by this device.
   */
  #[cfg(feature="alloc_ftr")]
  pub fn on_click(&self, bf: Box<dyn FnMut(PinIdentity, ButtonState) -> () + 'b>) {
    self.on_click.replace(Some(bf));
  }

  /**
   * Enable or disable the internal pullup on this pin
   */
  pub fn pullup(self, enable: bool) -> Self {
    match enable {
      true => self.pin.set_mode(PinMode::InputPullup),
      false => self.pin.set_mode(PinMode::InputFloating)
    }
    self
  }

  pub fn is_pressed(&self) -> bool {
    !self.pin.get()
  }

  pub fn state(&self) -> ButtonState {
    match self.pin.get() {
      true => ButtonState::Pressed,
      false => ButtonState::Released
    }
  }
}

impl<P,S> EventSource for Button<'_,P,S>
where
  P: Pin,
  S: EventSink
{
  fn listen(&'static self) {
    self.pin.listen(IsrCallback::WithData(|_source, id, state, udata|{
      S::event(OxideEventEnvelope::to(unsafe { &*(panic_if_none!(udata) as *const Button<P,S> as *const dyn EventSource) },
                                      OxideEvent::ButtonEvent(id,
                                                              match state {
                                                                true  => ButtonState::Pressed,
                                                                false => ButtonState::Released
                                                              })));
    }, self as *const dyn core::any::Any ))
  }

  fn process_event(&self, evt: OxideEvent) {
    match (self.on_click.borrow_mut().deref_mut(), evt) {
      (Some(f), OxideEvent::ButtonEvent(source, state)) => {
        (*f)(source,state)
      },
      _ => {}
    }
  }
}

