/* boot.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Boot process for the Oxide operating system.  Initialises memory and
//! interrupt vectors, loads our hardware table, and then calls the
//! oxide_main() method provided by whoever is using our crate.

// Imports ===================================================================
use crate::hal::concurrency;
use crate::hal::concurrency::thread;
use crate::deviceconsts::clock;
use crate::hal::generic::cpu::ClockControl;
#[cfg(feature="alloc_ftr")]
use crate::alloc;

// Declarations ==============================================================
extern {
  fn __oxide_main() -> !;
}

// Code ======================================================================
#[no_mangle]
fn _oxide_boot() {
  unsafe {
    // If we have a global allocator, initialise it
    #[cfg(feature="alloc_ftr")]
    alloc::initialise();

    // Inititalise the threading/concurrency system
    concurrency::internal::initialise();

    // Set up any hardware-specific configuration
    #[cfg(feature="atmega4809")]
    crate::hal::atmega4809::cpu::clock().clk_per_prescaler(clock::MASTER_CLOCK_PRESCALER);

    // Create the supervisor
    let mut supervisor = crate::oxide::create_supervisor();
    crate::oxide::set_global_supervisor(&mut supervisor);

    // Spawn the main thread
    let _jh = thread::spawn_threadunsafe(||{
      __oxide_main();
    });

    // Enter the scheduler loop
    concurrency::internal::run_scheduler();
  }
}
