/* event.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Oxide system events

// Imports ===================================================================
use crate::private::ringq::{Coalesce, QueueError};
use crate::devices::button::ButtonState;
use crate::devices::serialport::SerialState;
use crate::hal::generic::timer::TimerIdentity;
use crate::hal::generic::port::PinIdentity;
use crate::hal::generic::serial::SerialPortIdentity;

// Declarations ==============================================================
//pub type OxideEventQueue = RingQ<OxideEvent,16>;


/**
 * Events that can cause your application to wake up and need to do something.
 */
#[derive(Clone,Copy,PartialEq,Eq)]
pub enum OxideEvent {
  /**
   * Sent when the system starts up to allow the application to process any
   * initialisation tasks it needs.  This is guaranteed to be the first
   * event received by the application program.
   */
  Initialise,

  /**
   * Sent when a clock timer tick interrupt occurs.  The number
   * of elapsed clock ticks is included.
   */
  ClockTick(TimerIdentity, u16),

  /**
   * Sent when a button event occurs.  The state of the button when the
   * event was generated is included.
   */
  ButtonEvent(PinIdentity, ButtonState),

  /**
   * Sent when a serial port event occurs.
   */
  SerialEvent(SerialPortIdentity, SerialState)
}

/**
 * An envelope with a recipient address that we can wrap up Oxide events in.
 */
#[derive(Clone,Copy)]
pub struct OxideEventEnvelope<'e> {
  /**
   * What EventSource should process this event
   */
  receiver: Option<&'e dyn EventSource>,

  /**
   * The event itself
   */
  event: OxideEvent
}

/**
 * An event source is something which generates events for later processing.
 */
pub trait EventSource {
  fn listen(&'static self);

  fn process_event(&self, evt: OxideEvent);
}

/**
 * An event sink is something that devices can call to dump their events into.
 */
pub trait EventSink {
  fn event(event: OxideEventEnvelope);
}

/**
 * A 'null' event sunk that allows you to instantiate Oxide devices that
 * discard any generated events.
 */
pub struct EventDevNull {
}

// Code ======================================================================
impl<'e> OxideEventEnvelope<'e> {
  /**
   * Wrap an event intended for the given EventSource to process
   */
  pub fn to(receiver: &'e dyn EventSource, event: OxideEvent) -> Self {
    OxideEventEnvelope {
      receiver: Some(receiver),
      event
    }
  }

  /**
   * Wrap an event addressed to nobody in particular
   */
  pub fn anon(event: OxideEvent) -> Self {
    OxideEventEnvelope {
      receiver: None,
      event
    }
  }

  /**
   * Return the contents of the envelope (i.e. the event!)
   */
  pub fn unwrap(&self) -> OxideEvent {
    self.event
  }

  /**
   * Invoke the recipient's `process_event()` method (or just ignore it
   * if the event is unaddressed.)
   */
  pub(crate) fn invoke_recipient(&self) {
    match self.receiver {
      Some(ev_src) => {
        ev_src.process_event(self.event);
      },
      None => {}
    }
  }
}

impl Coalesce for OxideEvent {
  fn coalesce(&mut self, with: &Self) -> Result<(), QueueError> {
    match self {
      // MasterClockEvents we coalesce by adding the ticks together
      OxideEvent::ClockTick(source1, myticks) => {
        match with {
          OxideEvent::ClockTick(source2, ticks) => {
            if (source1 == source2) && (u16::MAX - *myticks) > *ticks {
              *myticks += ticks;
              Ok(())
            } else {
              Err(QueueError::CannotCoalesce)
            }
          },
          _ => Err(QueueError::CannotCoalesce)
        }
      },
      // Serial events we coalesce by just discarding duplicates
      OxideEvent::SerialEvent(source1, state) => {
        match with {
          OxideEvent::SerialEvent(source2, with_state) => {
            if (source1 == source2) && (state == with_state) {
              Ok(())
            } else {
              Err(QueueError::CannotCoalesce)
            }
          },
          _ => Err(QueueError::CannotCoalesce)
        }
      },
      _ => Err(QueueError::CannotCoalesce)
    }
  }
}

impl Coalesce for OxideEventEnvelope<'_> {
  fn coalesce(&mut self, with: &Self) -> Result<(), QueueError> {
    match (self.receiver, with.receiver) {
      (Some(my_receiver),Some(with_receiver)) => {
        if my_receiver as *const _ == with_receiver as *const _ {
          self.event.coalesce(&with.event)
        } else {
          Err(QueueError::CannotCoalesce)
        }
      }
      _=> Err(QueueError::CannotCoalesce)
    }
  }
}

/**
 * Do-nothing implementation of EventSink.
 */
impl EventSink for EventDevNull {
  fn event(_event: OxideEventEnvelope) {
  }
}

// Tests =====================================================================
#[cfg(test)]
mod tests {
  use crate::event::{EventSource, OxideEvent, OxideEventEnvelope};
  use crate::private::ringq::Coalesce;
  use crate::event::OxideEvent::{ButtonEvent, ClockTick, SerialEvent};
  use crate::devices::serialport::SerialState;
  use crate::devices::button::ButtonState;
  use crate::hal::generic::timer::TimerIdentity;
  use crate::hal::generic::serial::SerialPortIdentity;
  use crate::hal::generic::port::PinIdentity;
  use crate::panic_if_err;

  struct DummyEventSource {
  }
  impl EventSource for DummyEventSource {
    fn listen(&'static self) {
      todo!()
    }

    fn process_event(&self, evt: OxideEvent) {
      unimplemented!()
    }
  }

  #[test]
  fn test_event_coalesce_masterclockevent() {
    static DES : DummyEventSource = DummyEventSource {};

    let mut event1 = OxideEventEnvelope::to(&DES, ClockTick(TimerIdentity::Tcb0, 12));
    let event2 = OxideEventEnvelope::to(&DES, ClockTick(TimerIdentity::Tcb0, 14));

    event1.coalesce(&event2);

    if let ClockTick(TimerIdentity::Tcb0, ticks) = event1.unwrap() {
      println!("Coalesced MasterClockEvent (ticks == {})", ticks);
      assert_eq!(ticks, 26);
    }
  }

  #[test]
  fn test_event_coalesce_serialevent_ok() {
    static DES : DummyEventSource = DummyEventSource {};

    let mut event1 = OxideEventEnvelope::to(&DES, SerialEvent(SerialPortIdentity::Usart0, SerialState::ReadAvailable));
    let event2 = OxideEventEnvelope::to(&DES, SerialEvent(SerialPortIdentity::Usart0, SerialState::ReadAvailable));

    panic_if_err!(event1.coalesce(&event2));

    let mut event1 = OxideEventEnvelope::to(&DES, SerialEvent(SerialPortIdentity::Usart0, SerialState::BreakDetected));
    let event2 = OxideEventEnvelope::to(&DES, SerialEvent(SerialPortIdentity::Usart0, SerialState::BreakDetected));

    panic_if_err!(event1.coalesce(&event2));
  }

  #[test]
  #[should_panic]
  fn test_event_coalesce_serialevent_fail() {
    static DES : DummyEventSource = DummyEventSource {};

    let mut event1 = OxideEventEnvelope::to(&DES, SerialEvent(SerialPortIdentity::Usart0, SerialState::ReadAvailable));
    let event2 = OxideEventEnvelope::to(&DES, SerialEvent(SerialPortIdentity::Usart0, SerialState::BreakDetected));

    let thing = event1.coalesce(&event2);

    panic_if_err!(event1.coalesce(&event2));
  }

  #[test]
  #[should_panic]
  fn test_event_coalesce_fail() {
    static DES : DummyEventSource = DummyEventSource {};

    let mut event1 = OxideEventEnvelope::to(&DES, ClockTick(TimerIdentity::Tcb0, 42));
    let event2 = OxideEventEnvelope::to(&DES, ButtonEvent(PinIdentity::PortA(0),ButtonState::Pressed));

   panic_if_err!(event1.coalesce(&event2));
  }
}