/* mod.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Higher-level device abstractions for things like LEDs, system clocks,
//! buttons.
use crate::devices::button::Button;
use crate::devices::led::Led;
use crate::devices::masterclock::MasterClock;
use crate::devices::serialport::SerialPort;
use crate::devices::wallclock::WallClock;
use crate::oxide::OxideSupervisor;

pub mod masterclock;
pub mod wallclock;
pub mod led;
pub mod button;
pub mod serialport;
pub mod debouncer;

/// A simple button attached to a hardware pin that generates events with the
/// standard AVRoxide Supervisor implementation.
pub type OxideButton<'a,'b,P> = Button<'a,P,OxideSupervisor<'b>>;
/// A simple LED attached to a hardware pin.
pub type OxideLed<P> = Led<P>;
/// A high-frequency, low-accuracy clock event source that works with the
/// standard AVRoxide Supervisor implementation.
pub type OxideMasterClock<'a,'b,T> = MasterClock<'a,T,OxideSupervisor<'b>>;
/// A simple serial port device that works with the standard AVRoxide
/// Supervisor implementation.
pub type OxideSerialPort<'a,'s,P,T,R> = SerialPort<'a,P,T,R,OxideSupervisor<'s>>;
/// A low-frequency, high-accuracy clock event source that works with the
/// standard AVRoxide Supervisor implementation.
pub type OxideWallClock<'a,'b,T> = WallClock<'a,T,OxideSupervisor<'b>>;
