/* led.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Simple abstraction of an LED attached to a GPIO pin.

// Imports ===================================================================
use crate::hal::generic::port::{Pin, PinMode, InterruptMode};
use crate::util::OwnOrBorrowMut;

// Declarations ==============================================================
pub struct Led<P>
where
  P: 'static + Pin
{
  pin: OwnOrBorrowMut<'static,P>
}

// Code ======================================================================
impl<P> Led<P>
where
  P: Pin
{
  pub fn using<OP: Into<OwnOrBorrowMut<'static, P>>>(pin: OP) -> Self {
    let pin : OwnOrBorrowMut<P> = pin.into();

    pin.set_mode(PinMode::Output);
    pin.set_interrupt_mode(InterruptMode::Disabled);
    Led { pin }
  }

  pub fn with_pin(pin: &'static mut P) -> Self {
    Self::using(pin)
  }

  pub fn set_on(&self) {
    self.pin.set_high();
  }

  pub fn set_off(&self) {
    self.pin.set_low();
  }

  pub fn set(&self, on: bool) {
    match on {
      true => self.set_on(),
      false => self.set_off()
    }
  }

  pub fn toggle(&self) {
    self.pin.toggle()
  }
}
