/* arduino.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! A helper module that maps standard ATmega pins into equivalent Arduino
//! names.


// Imports ===================================================================

// Declarations ==============================================================
#[cfg(feature="atmega4809")]
pub mod nanoevery {
  use crate::hal::atmega4809::hardware::Hardware;
  use crate::hal::atmega4809;
  use crate::hal::generic::port::PinIdentity;

  /**
   * Arduino-style pin identifier, used to identify a pin event source at
   * runtime
   */
  pub enum ArduinoPinIdentity {
    /// The so-called 'analogue' pins (that is, pins that can be attached to
    /// a timer's PWM output)
    PinA(u8),
    /// The 'digital' pins
    PinD(u8)
  }

  pub struct Arduino {
    #[cfg(feature="tcb0")]
    pub timer0: &'static mut atmega4809::timer::tcb0::TimerImpl,
    #[cfg(feature="tcb1")]
    pub timer1: &'static mut atmega4809::timer::tcb1::TimerImpl,
    #[cfg(feature="tcb2")]
    pub timer2: &'static mut atmega4809::timer::tcb2::TimerImpl,
    #[cfg(feature="tcb3")]
    pub timer3: &'static mut atmega4809::timer::tcb3::TimerImpl,

    pub a0: &'static mut atmega4809::port::portd::PinImpl,
    pub a1: &'static mut atmega4809::port::portd::PinImpl,
    pub a2: &'static mut atmega4809::port::portd::PinImpl,
    pub a3: &'static mut atmega4809::port::portd::PinImpl,
    pub a4: &'static mut atmega4809::port::porta::PinImpl,
    pub a5: &'static mut atmega4809::port::porta::PinImpl,
    pub a6: &'static mut atmega4809::port::portd::PinImpl,
    pub a7: &'static mut atmega4809::port::portd::PinImpl,

    pub d0: &'static mut atmega4809::port::portc::PinImpl,
    pub d1: &'static mut atmega4809::port::portc::PinImpl,
    pub d2: &'static mut atmega4809::port::porta::PinImpl,
    pub d3: &'static mut atmega4809::port::portf::PinImpl,
    pub d4: &'static mut atmega4809::port::portc::PinImpl,
    pub d5: &'static mut atmega4809::port::portb::PinImpl,
    pub d6: &'static mut atmega4809::port::portf::PinImpl,
    pub d7: &'static mut atmega4809::port::porta::PinImpl,
    pub d8: &'static mut atmega4809::port::porte::PinImpl,
    pub d9: &'static mut atmega4809::port::portb::PinImpl,
    pub d10: &'static mut atmega4809::port::portb::PinImpl,
    pub d11: &'static mut atmega4809::port::porte::PinImpl,
    pub d12: &'static mut atmega4809::port::porte::PinImpl,
    pub d13: &'static mut atmega4809::port::porte::PinImpl,

    pub watchdog: &'static mut atmega4809::watchdog::WatchdogImpl,

    #[cfg(feature="usart3")]
    pub usb_serial: &'static mut atmega4809::serial::usart3::SerialImpl,
    #[cfg(feature="usart3")]
    pub usb_serial_tx: &'static mut atmega4809::port::portb::PinImpl,
    #[cfg(feature="usart3")]
    pub usb_serial_rx: &'static mut atmega4809::port::portb::PinImpl,
  }


  impl From<&'static mut crate::hal::atmega4809::hardware::Hardware> for Arduino {
    fn from(avr: &'static mut Hardware) -> Self {
      Arduino {
        #[cfg(feature="tcb0")]
        timer0: avr.timerb0,
        #[cfg(feature="tcb1")]
        timer1: avr.timerb1,
        #[cfg(feature="tcb2")]
        timer2: avr.timerb2,
        #[cfg(feature="tcb3")]
        timer3: avr.timerb3,

        a0: atmega4809::port::portd::pin_instance(3),
        a1: atmega4809::port::portd::pin_instance(2),
        a2: atmega4809::port::portd::pin_instance(1),
        a3: atmega4809::port::portd::pin_instance(0),
        a4: atmega4809::port::porta::pin_instance(2),
        a5: atmega4809::port::porta::pin_instance(3),
        a6: atmega4809::port::portd::pin_instance(4),
        a7: atmega4809::port::portd::pin_instance(5),
        d0: atmega4809::port::portc::pin_instance(4),
        d1: atmega4809::port::portc::pin_instance(5),
        d2: atmega4809::port::porta::pin_instance(0),
        d3: atmega4809::port::portf::pin_instance(5),
        d4: atmega4809::port::portc::pin_instance(6),
        d5: atmega4809::port::portb::pin_instance(2),
        d6: atmega4809::port::portf::pin_instance(4),
        d7: atmega4809::port::porta::pin_instance(1),
        d8: atmega4809::port::porte::pin_instance(3),
        d9: atmega4809::port::portb::pin_instance(0),
        d10: atmega4809::port::portb::pin_instance(1),
        d11: atmega4809::port::porte::pin_instance(0),
        d12: atmega4809::port::porte::pin_instance(1),
        d13: atmega4809::port::porte::pin_instance(2),

        watchdog: avr.watchdog,

        #[cfg(feature="usart3")]
        usb_serial: avr.usart3.mux(atmega4809::serial::usart3::SerialPortPins::PortB54),
        #[cfg(feature="usart3")]
        usb_serial_tx: atmega4809::port::portb::pin_instance(4),
        #[cfg(feature="usart3")]
        usb_serial_rx: atmega4809::port::portb::pin_instance(5),
      }
    }
  }

  impl From<PinIdentity> for ArduinoPinIdentity {
    fn from(id: PinIdentity) -> Self {
      match id {
        PinIdentity::PortA(0) => ArduinoPinIdentity::PinD(2),
        PinIdentity::PortA(1) => ArduinoPinIdentity::PinD(7),
        PinIdentity::PortA(2) => ArduinoPinIdentity::PinA(4),
        PinIdentity::PortA(3) => ArduinoPinIdentity::PinA(5),

        PinIdentity::PortB(0) => ArduinoPinIdentity::PinD(9),
        PinIdentity::PortB(1) => ArduinoPinIdentity::PinD(10),
        PinIdentity::PortB(2) => ArduinoPinIdentity::PinD(5),

        PinIdentity::PortC(4) => ArduinoPinIdentity::PinD(0),
        PinIdentity::PortC(5) => ArduinoPinIdentity::PinD(1),
        PinIdentity::PortC(6) => ArduinoPinIdentity::PinD(4),

        PinIdentity::PortD(0) => ArduinoPinIdentity::PinA(3),
        PinIdentity::PortD(1) => ArduinoPinIdentity::PinA(2),
        PinIdentity::PortD(2) => ArduinoPinIdentity::PinA(1),
        PinIdentity::PortD(3) => ArduinoPinIdentity::PinA(0),
        PinIdentity::PortD(4) => ArduinoPinIdentity::PinA(6),
        PinIdentity::PortD(5) => ArduinoPinIdentity::PinA(7),

        PinIdentity::PortE(3) => ArduinoPinIdentity::PinD(8),
        PinIdentity::PortE(0) => ArduinoPinIdentity::PinD(11),
        PinIdentity::PortE(1) => ArduinoPinIdentity::PinD(12),
        PinIdentity::PortE(2) => ArduinoPinIdentity::PinD(13),

        PinIdentity::PortF(5) => ArduinoPinIdentity::PinD(3),
        PinIdentity::PortF(4) => ArduinoPinIdentity::PinD(6),

        _ => panic!()
      }
    }
  }
}
// Tests =====================================================================
