/* ports.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Generic I/O ports provided by the AVR microcontroller.

// Imports ===================================================================



// Declarations ==============================================================
/**
 * Input/output mode for a pin
 */
#[derive(Clone,Copy,PartialEq,Eq)]
pub enum PinMode {
  /// Pin is configured as output
  Output,
  /// Pin is configured as an input with internal pullup enabled
  InputPullup,
  /// Pin is configured as an input with internal pullup disabled
  InputFloating
}

/**
 * Interrupt generation mode for a pin
 */
#[derive(Clone,Copy,PartialEq,Eq)]
pub enum InterruptMode {
  /// Do not generate interrupts
  Disabled,
  /// Interrupt on both rising and falling edges
  BothEdges,
  /// Interrupt on rising edge
  RisingEdge,
  /// Interrupt on falling edge
  FallingEdge,
  /// Interrupt while input is low
  LowLevel
}

/**
 * Callback called by a pin when it generates an interrupt.  The callback
 * is given the new state read from the pin (true = high).
 * The callback should return a boolean:
 * * true:  Continue generating interrupts for this pin
 * * false: Disable generating interrupts for this pin
 *
 * Note: The callback runs *within the interrupt context* - be careful about
 * using mutual exclusion where necessary, and *DO NOT DO HEAVY PROCESSING
 * IN THE CALLBACK*.
 */
pub type PinInterruptHandler = fn(bool) -> ();

pub trait Pin {
  /// Set the input/output mode for this pin
  fn set_mode(&mut self, mode: PinMode);

  /// Toggle this pin's output level
  fn toggle(&mut self);

  /// Set the pin's output to high
  fn set_high(&mut self);

  /// Set the pin's output to low
  fn set_low(&mut self);

  /// Set the pin's output according to the boolean (true == high, false == low)
  fn set(&mut self, high: bool);

  /// Get the pin's input; true == high, false == low.
  fn get(&self) -> bool;

  /// Set when this pin will generate interrupts
  fn set_interrupt_mode(&mut self, mode: InterruptMode);

  /// Listen for interrupts on this pin and call the given handler callback
  /// when one occurs.
  fn listen(&mut self, handler: Option<PinInterruptHandler>);

}

pub trait Port<P>
where
  P: Pin
{
  fn pin_instance(&self, pin: u8) -> &P;
}

// Code ======================================================================
#[cfg(target_arch="avr")]
pub mod base {
  use crate::hal::generic::port::InterruptMode;

  pub trait AtmelPortControl {
    /// Set this pin as an output.
    fn enable_output(&mut self, p: u8);
    /// Disable output on this pin
    fn disable_output(&mut self, p: u8);
    /// Enable internal pullup on this pin
    fn enable_pullup(&mut self, p: u8);
    /// Disable internal pullup on this pin
    fn disable_pullup(&mut self, p: u8);
    /// Set a pin's output state high
    fn set_high(&mut self, p: u8);
    /// Set a pin's output state low
    fn set_low(&mut self, p: u8);
    /// Toggle a pin's input state
    fn toggle(&mut self, p: u8);
    /// Get a pin's state
    fn get(&self, p: u8) -> bool;
    /// Set the mode for interrupt generation for a pin
    fn set_interrupt_mode(&mut self, p: u8, mode: InterruptMode);
    /// True iff the given pin has generated an interrupt
    fn interrupted(&self, p: u8) -> bool;
    /// Clear the interrupt flag for all pins on this port
    fn clear_interrupts(&mut self);
  }

  impl InterruptMode {
    #[inline(always)]
    fn to_pinctrl_isc(&self) -> u8 {
      match self {
        InterruptMode::Disabled => 0x00,
        InterruptMode::BothEdges => 0x01,
        InterruptMode::RisingEdge => 0x02,
        InterruptMode::FallingEdge => 0x03,
        InterruptMode::LowLevel => 0x05
      }
    }

    #[inline(always)]
    fn pinctrl_mask() -> u8 {
      0b00000111
    }
  }

  /**
   * The AVR port control register block.  The structure provided by the
   * auto-generated avr-device crate is just horrible, so we use our own
   * rendition.  This allows accessing pin control by index, and also avoids
   * having different types for every port.
   */
  #[repr(C)]
  pub struct AvrPortRegisterBlock {
    pub(crate) dir: u8,
    pub(crate) dir_set: u8,
    pub(crate) dir_clr: u8,
    pub(crate) dir_tgl: u8,
    pub(crate) out: u8,
    pub(crate) out_set: u8,
    pub(crate) out_clr: u8,
    pub(crate) out_tgl: u8,
    pub(crate) inp: u8,
    pub(crate) intflags: u8,
    pub(crate) portctrl: u8,
    pub(crate) reserved: [u8; 5],
    pub(crate) pinctrl: [u8; 8]
  }



  impl AtmelPortControl for AvrPortRegisterBlock {
    #[inline(always)]
    fn enable_output(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.dir_set as *mut u8, 0x01 << p);
      }
    }
    #[inline(always)]
    fn disable_output(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.dir_clr as *mut u8, 0x01 << p);
      }
    }

    fn enable_pullup(&mut self, p: u8) {
      unsafe {
        let pinctrl_reg = &mut self.pinctrl[p as usize] as *mut u8;

        let pinctrl = core::ptr::read_volatile(pinctrl_reg);

        core::ptr::write_volatile(pinctrl_reg, pinctrl | 0b00001000);
      }
    }

    fn disable_pullup(&mut self, p: u8) {
      unsafe {
        let pinctrl_reg = &mut self.pinctrl[p as usize] as *mut u8;

        let pinctrl = core::ptr::read_volatile(pinctrl_reg);

        core::ptr::write_volatile(pinctrl_reg, pinctrl & 0b11110111);
      }
    }

    #[inline(always)]
    fn set_high(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.out_set as *mut u8, 0x01 << p);
      }
    }
    #[inline(always)]
    fn set_low(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.out_clr as *mut u8, 0x01 << p);
      }
    }
    #[inline(always)]
    fn toggle(&mut self, p: u8) {
      unsafe {
        core::ptr::write_volatile(&mut self.out_tgl as *mut u8, 0x01 << p);
      }
    }

    #[inline(always)]
    fn get(&self, p: u8) -> bool {
      unsafe {
        core::ptr::read_volatile(&self.inp as *const u8) & (0x01 << p) != 0
      }
    }

    fn set_interrupt_mode(&mut self, p: u8, mode: InterruptMode) {
      unsafe {
        let pinctrl_reg = &mut self.pinctrl[p as usize] as *mut u8;

        let pinctrl = core::ptr::read_volatile(pinctrl_reg);

        core::ptr::write_volatile(pinctrl_reg,
                                  (pinctrl & !InterruptMode::pinctrl_mask()) | mode.to_pinctrl_isc());
      }
    }

    #[inline(always)]
    fn interrupted(&self, p: u8) -> bool {
      unsafe {
        core::ptr::read_volatile(&self.intflags as *const u8) & (0x01 << p) != 0
      }
    }

    #[inline(always)]
    fn clear_interrupts(&mut self) {
      unsafe {
        core::ptr::write_volatile(&mut self.intflags as *mut u8, 0xff);
      }
    }


  }

  #[macro_export]
  macro_rules! atmel_port_tpl {
    ($ref:expr, $isr:ident) => {


      use crate::hal::generic::port::{PinMode,Pin,InterruptMode,PinInterruptHandler};
      use crate::hal::generic::port::base::{AtmelPortControl,AvrPortRegisterBlock};

      pub type PortImpl = AvrPortRegisterBlock;
      pub type PinImpl = AtmelPin;

      /**
       * A no-op function that is the default handler for the button
       * interrupt callbacks.
       */
      #[inline(never)]
      fn nop_handler(_state: bool) -> () {
        unsafe { llvm_asm!("nop" :::: "volatile" ) }
      }

      /**
       * A single pin instance.  This is just a wrapper for a reference to
       * the pin number and a place to stash the callback function that will
       * be called when we have an interrupt.
       */
      pub struct AtmelPin {
        n: u8,
        handler: PinInterruptHandler
      }

      /**
       * Implementation of the Pin API for our Atmel PINs.
       */
      impl Pin for AtmelPin {
        #[inline(always)]
        fn set_mode(&mut self, mode: PinMode) {
          match mode {
            PinMode::Output => {
              instance().enable_output(self.n);
            },
            PinMode::InputPullup => {
              instance().enable_pullup(self.n);
              instance().disable_output(self.n);
            },
            PinMode::InputFloating => {
              instance().disable_pullup(self.n);
              instance().disable_output(self.n);
            }
          }
        }
        #[inline(always)]
        fn toggle(&mut self) {
          instance().toggle(self.n);
        }
        #[inline(always)]
        fn set_high(&mut self) {
          instance().set_high(self.n);
        }
        #[inline(always)]
        fn set_low(&mut self) {
          instance().set_low(self.n);
        }
        #[inline(always)]
        fn set(&mut self, high: bool) {
          match high {
            true  => instance().set_high(self.n),
            false => instance().set_low(self.n)
          }
        }
        #[inline(always)]
        fn get(&self) -> bool {
          instance().get(self.n)
        }
        fn set_interrupt_mode(&mut self, mode: InterruptMode) {
          instance().set_interrupt_mode(self.n, mode)
        }
        fn listen(&mut self, handler: Option<PinInterruptHandler>){
          crate::hal::concurrency::interrupt::isolated(||{
            unsafe {
              PINS[self.n as usize].handler = match handler {
                Some(handler) => handler,
                None => nop_handler
              }
            }
          });
        }
      }

      static mut INITIALISED: bool = false;

      /**
       * Get an instance of this port's register block.  Also does any
       * static initialisation required of the device the first time it is
       * called.
       */
      #[inline(always)]
      pub fn instance() -> &'static mut AvrPortRegisterBlock  {
        unsafe {
          // Not sure if this is a compiler bug or what, but our static
          // muts don't seem to be correctly initialised, so we need to
          // do this here instead
          //
          // ( @todo investigate - Probably I need to amend my boot code to call
          //         something to do the initialisation for me )
          if(!INITIALISED){
            for pin in 0..=7 {
              PINS[pin].n = pin as u8;
              PINS[pin].handler = nop_handler;
            }
            INITIALISED = true;
          }

          core::mem::transmute($ref)
        }
      }

      static mut PINS : [AtmelPin; 8] = [
        AtmelPin { n: 0, handler: nop_handler },
        AtmelPin { n: 1, handler: nop_handler },
        AtmelPin { n: 2, handler: nop_handler },
        AtmelPin { n: 3, handler: nop_handler },
        AtmelPin { n: 4, handler: nop_handler },
        AtmelPin { n: 5, handler: nop_handler },
        AtmelPin { n: 6, handler: nop_handler },
        AtmelPin { n: 7, handler: nop_handler },
      ];


      /**
       * Return a single pin instance wrapper.
       */
      pub fn pin_instance(pin: u8) -> &'static mut AtmelPin {
        debug_assert!(pin < 8);

        unsafe {
          PINS[0].handler = nop_handler;

          &mut PINS[pin as usize]
        }
      }

      /**
       * Interrupt service routine.  Checks which pins caused the interrupt
       * and then calls the relevant callback routine.
       */
      #[no_mangle]
      pub unsafe extern "avr-interrupt" fn $isr() {
        crate::hal::concurrency::interrupt::isr(||{
          let intflags:u8 = core::ptr::read_volatile(&instance().intflags as *const u8);
          let state:u8    = core::ptr::read_volatile(&instance().inp as *const u8);
          let mut mask    = 0x01;

          for pin in 0..=7 {
            if (intflags & mask) > 0 {
              (PINS[pin].handler)((state & mask) > 0);
            }
            mask <<= 1;
          }
          instance().clear_interrupts();
        })
      }
    }
  }
}

#[cfg(not(target_arch="avr"))]
pub mod base {
  use crate::hal::generic::port::InterruptMode;

  pub trait AtmelPortControl {
    /// Set this pin as an output.
    fn enable_output(&mut self, p: u8);
    /// Disable output on this pin
    fn disable_output(&mut self, p: u8);
    /// Enable internal pullup on this pin
    fn enable_pullup(&mut self, p: u8);
    /// Disable internal pullup on this pin
    fn disable_pullup(&mut self, p: u8);
    /// Set a pin's output state high
    fn set_high(&mut self, p: u8);
    /// Set a pin's output state low
    fn set_low(&mut self, p: u8);
    /// Toggle a pin's input state
    fn toggle(&mut self, p: u8);
    /// Get a pin's state
    fn get(&self, p: u8) -> bool;
    /// Set the mode for interrupt generation for a pin
    fn set_interrupt_mode(&mut self, p: u8, mode: InterruptMode);
    /// True iff the given pin has generated an interrupt
    fn interrupted(&self, p: u8) -> bool;
    /// Clear the interrupt flag for all pins on this port
    fn clear_interrupts(&mut self);
  }

  /**
   * The AVR port control register block.  The structure provided by the
   * auto-generated avr-device crate is just horrible, so we use our own
   * rendition.  This allows accessing pin control by index, and also avoids
   * having different types for every port.
   */
  #[repr(C)]
  pub struct DummyPortRegisterBlock {
    pub(crate) pin: [bool; 8],
    pub(crate) o_en: [bool; 8],
    pub(crate) pu_en: [bool; 8]
  }



  impl AtmelPortControl for DummyPortRegisterBlock {
    fn enable_output(&mut self, p: u8) {
      println!("*** PORT: Enable output pin {}", p);
      self.o_en[p as usize] = true;
    }

    fn disable_output(&mut self, p: u8) {
      println!("*** PORT: Disable output pin {}", p);
      self.o_en[p as usize] = false;
    }

    fn enable_pullup(&mut self, p: u8) {
      println!("*** PORT: Enable pullup pin {}", p);
      self.pu_en[p as usize] = true;
    }

    fn disable_pullup(&mut self, p: u8) {
      println!("*** PORT: Disable pullup pin {}", p);
      self.pu_en[p as usize] = false;
    }

    fn set_high(&mut self, p: u8) {
      println!("*** PORT: Set HIGH pin {}", p);
      self.pin[p as usize] = true;
    }

    fn set_low(&mut self, p: u8) {
      println!("*** PORT: Set LOW pin {}", p);
      self.pin[p as usize] = false;
    }

    fn toggle(&mut self, p: u8) {
      let old = self.pin[p as usize];

      println!("*** PORT: TOGGLE pin {} ({} => {})", p, old, !old);

      self.pin[p as usize] = !old;
    }

    fn get(&self, p: u8) -> bool {
      let val = self.pin[p as usize];
      println!("*** PORT: GET pin {} ({})", p, val);
      val
    }

    fn set_interrupt_mode(&mut self, p: u8, _mode: InterruptMode) {
      println!("*** PORT: Set interrupt mode pin {}", p);
    }

    fn interrupted(&self, p: u8) -> bool {
      println!("*** PORT: Get interrupt flag pin {}", p);
      false
    }

    fn clear_interrupts(&mut self) {
      println!("*** PORT: Clear all interrupts");
    }


  }

  #[macro_export]
  macro_rules! atmel_port_tpl {
    ($ref:expr, $isr:ident) => {


      use crate::hal::generic::port::{PinMode,Pin,InterruptMode,PinInterruptHandler};
      use crate::hal::generic::port::base::{AtmelPortControl,DummyPortRegisterBlock};

      pub type PortImpl = DummyPortRegisterBlock;
      pub type PinImpl = AtmelPin;

      /**
       * A single pin instance.  This is just a wrapper for a reference to
       * the pin number and a place to stash the callback function that will
       * be called when we have an interrupt.
       */
      pub struct AtmelPin {
        n: u8,
        handler: PinInterruptHandler
      }

      /**
       * A no-op function that is the default handler for the button
            * interrupt callbacks.
       */
      fn nop_handler(_state: bool) -> () {
      }

      /**
       * Implementation of the Pin API for our Atmel PINs.
       */
      impl Pin for AtmelPin {
        #[inline(always)]
        fn set_mode(&mut self, mode: PinMode) {
          match mode {
            PinMode::Output => {
              instance().enable_output(self.n);
            },
            PinMode::InputPullup => {
              instance().enable_pullup(self.n);
              instance().disable_output(self.n);
            },
            PinMode::InputFloating => {
              instance().disable_pullup(self.n);
              instance().disable_output(self.n);
            }
          }
        }
        #[inline(always)]
        fn toggle(&mut self) {
          instance().toggle(self.n);
        }
        #[inline(always)]
        fn set_high(&mut self) {
          instance().set_high(self.n);
        }
        #[inline(always)]
        fn set_low(&mut self) {
          instance().set_low(self.n);
        }
        #[inline(always)]
        fn set(&mut self, high: bool) {
          match high {
            true  => instance().set_high(self.n),
            false => instance().set_low(self.n)
          }
        }
        #[inline(always)]
        fn get(&self) -> bool {
          instance().get(self.n)
        }
        fn set_interrupt_mode(&mut self, mode: InterruptMode) {
          instance().set_interrupt_mode(self.n, mode)
        }
        fn listen(&mut self, handler: Option<PinInterruptHandler>){
          crate::hal::concurrency::interrupt::isolated(||{
            unsafe {
              PINS[self.n as usize].handler = match handler {
                Some(handler) => handler,
                None => nop_handler
              }
            }
          });
        }
      }

      static mut INITIALISED: bool = false;

      static mut INSTANCE : DummyPortRegisterBlock = DummyPortRegisterBlock {
        pin: [false; 8],
        o_en: [false; 8],
        pu_en: [false; 8]
      };

      /**
       * Get an instance of this port's register block.  Also does any
       * static initialisation required of the device the first time it is
       * called.
       */
      #[inline(always)]
      pub fn instance() -> &'static mut DummyPortRegisterBlock  {
        unsafe {
          // Not sure if this is a compiler bug or what, but our static
          // muts don't seem to be correctly initialised, so we need to
          // do this here instead
          //
          // ( @todo investigate - Probably I need to amend my boot code to call
          //         something to do the initialisation for me )
          if(!INITIALISED){
            for pin in 0..=7 {
              PINS[pin].n = pin as u8;
              PINS[pin].handler = nop_handler;
            }
            INITIALISED = true;
          }

          &mut INSTANCE
        }
      }

      static mut PINS : [AtmelPin; 8] = [
        AtmelPin { n: 0, handler: nop_handler },
        AtmelPin { n: 1, handler: nop_handler },
        AtmelPin { n: 2, handler: nop_handler },
        AtmelPin { n: 3, handler: nop_handler },
        AtmelPin { n: 4, handler: nop_handler },
        AtmelPin { n: 5, handler: nop_handler },
        AtmelPin { n: 6, handler: nop_handler },
        AtmelPin { n: 7, handler: nop_handler },
      ];


      /**
       * Return a single pin instance wrapper.
       */
      pub fn pin_instance(pin: u8) -> &'static mut AtmelPin {
        debug_assert!(pin < 8);

        unsafe {
          PINS[0].handler = nop_handler;

          &mut PINS[pin as usize]
        }
      }
    }
  }
}