/* oxide.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! A very basic "operating system" (ha!) for our embedded devices, that
//! essentially comprises an event loop that executes a provided closure
//! whenever something happens.  "Something happens" is indicated by
//! sending events to the supervisor.

// Imports ===================================================================
use crate::ringq::RingQ;
use crate::event::{OxideEvent, EventSink};
use crate::mut_singleton;

// Declarations ==============================================================

pub struct OxideSupervisor
{
  event_q: RingQ<OxideEvent, 16>
}

mut_singleton!(OxideSupervisor, INSTANCE, instance,
  OxideSupervisor {
    event_q: RingQ::new()
  });

// Code ======================================================================
impl OxideSupervisor {
  /**
   * Enter the event loop - and never return (*evil cackle*)
   */
  pub fn run<F: FnMut(OxideEvent)>(&mut self, mut handler: F) -> ! {
    unsafe {
      crate::hal::concurrency::imp::enable_interrupts();
    }

    loop {
      crate::hal::concurrency::imp::wait_for_interrupt();

      while let Some(event) = self.event_q.consume() {
        handler(event);
      }
    }
  }
}

impl EventSink for OxideSupervisor {
  fn event(event: OxideEvent) -> () {
    instance().event_q.append(event).unwrap();
  }
}


// Tests =====================================================================
