/* masterclock.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */

// Imports ===================================================================
use crate::hal::generic::timer::TimerControl;
use crate::hal::generic::timer::TimerMode::Periodic;
use crate::event::{EventSink, OxideEvent};

use crate::deviceconsts::my::{ MASTER_CLOCK_PRESCALER, MASTER_CLOCK_HZ, MASTER_TICK_PERIOD_CS };
use core::marker::PhantomData;

// Declarations ==============================================================


pub struct MasterClock<T,S>
where
  T: 'static + TimerControl,
  S: EventSink
{
  timer: &'static mut T,
  phantom: PhantomData<S>
}


// Code ======================================================================

impl<T,S> MasterClock<T,S>
where
  T: TimerControl,
  S: EventSink
{
  pub fn using_timer(timer: &'static mut T) -> Self {
    const PERIPHERAL_CLOCK_HZ: u32  = MASTER_CLOCK_HZ/MASTER_CLOCK_PRESCALER as u32;
    const TIMER_CLOCK_HZ: u32       = PERIPHERAL_CLOCK_HZ/2; // We use CLK_PER/2 in timer config
    const CYCLES_PER_CS: u16        = (TIMER_CLOCK_HZ/100) as u16;

    Self {
      timer: timer.mode(Periodic).count_max(CYCLES_PER_CS).interrupting(MASTER_TICK_PERIOD_CS),
      phantom: PhantomData::default()
    }
  }

  pub fn run(&mut self) {
    self.timer.start(Some(|ticks:u16| {
      S::event(OxideEvent::MasterClockEvent(ticks));
      true
    }));
  }
}


// Tests =====================================================================
