/* timer.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Generic trait for controlling timer devices

// Imports ===================================================================

// Declarations ==============================================================
#[derive(Clone,Copy)]
pub enum TimerMode {
  Periodic
}

/**
 * Callback called by a timer when it generates an interrupt.  The callback
 * is given the number of ticks counted since the last such event.  The
 * callback should return a boolean:
 * * true:  Continue timer running
 * * false: Stop timer running
 *
 * Note: The callback runs *within the interrupt context* - be careful about
 * using mutual exclusion where necessary, and *DO NOT DO HEAVY PROCESSING
 * IN THE CALLBACK*.
 */
pub(crate) type TimerInterruptHandler = fn(u16) ->bool;

pub trait TimerControl {
  /// Builder method that sets the number of underlying timer events that
  /// trigger an interrupt callback (i.e. if 10, every 10 timer events
  /// the callback passed to start() will be called.)
  fn interrupting(&mut self, period: u16) -> &mut Self;

  /// Builder method that sets the clock's mode
  fn mode(&mut self, mode: TimerMode) -> &mut Self;

  /// Builder method that sets the clock's counter trigger
  fn count_max(&mut self, max: u16) -> &mut Self;

  /// Start this timer.  The given callback will be called periodically
  /// when timer interrupts occur (see the interrupting(period) method.)
  ///
  /// If the timer is in a constant-run mode (e.g. TimerMode::Periodic), it
  /// will run constantly until either stopped (with the stop() method!)
  /// or until the callback function returns false.
  fn start(&mut self, handler: Option<TimerInterruptHandler>);

  /// Stop this timer
  fn stop(&mut self);

  /// Get the timer's current count value
  fn get_count(&self) -> u16;

  /// Reset the timer's current count value
  fn reset_count(&mut self);
}


// Code ======================================================================


