/* supervisor.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! A very basic "operating system" (ha!) for our embedded devices, that
//! essentially comprises an event loop that executes a provided closure
//! whenever something happens.  "Something happens" is indicated by
//! sending events to the supervisor.

// Imports ===================================================================

// Declarations ==============================================================
pub const MASTER_CLOCK_HZ: u32       = 20_000_000; // Config'd in device fuses
pub const MAIN_CLOCK_PRESCALER:  u8  = 6;          // Device default (per datasheet)
pub const MASTER_TICK_PERIOD_CS: u16 = 5;          // Our desire - 20Hz


/**
 * Hardware that we can supervise needs to implement this trait.
 */
pub trait HardwareEventSource {
  type Event : Clone;

  fn enable_events(&mut self);

  fn disable_events(&mut self);

  fn wait_for_event(&mut self) -> Option<Self::Event>;
}


pub struct Supervisor<'s,H>
where
  H: HardwareEventSource
{
  hardware: &'s mut H
}



// Code ======================================================================
impl<'s,H> Supervisor<'s,H>
where
  H: HardwareEventSource
{
  pub fn for_hardware(hardware: &'s mut H) -> Self {
    Self {
      hardware
    }
  }

  /**
   * Enter the event loop - and never return (*evil cackle*)
   */
  pub fn run<F: FnMut(H::Event)>(&mut self, mut handler: F) -> ! {
    self.hardware.enable_events();

    loop {
      match self.hardware.wait_for_event() {
        Some(event) => handler(event),
        None => {}
      }
    }
  }


}


// Tests =====================================================================
