#[doc = r"Enumeration of all the interrupts."]
#[derive(Copy, Clone, Debug, PartialEq, Eq)]
#[repr(u16)]
pub enum Interrupt {
    #[doc = "0 - External Pin, Power-on Reset, Brown-out Reset and Watchdog Reset"]
    RESET = 0,
    #[doc = "1 - External Interrupt Request 0"]
    INT0 = 1,
    #[doc = "2 - External Interrupt Request 1"]
    INT1 = 2,
    #[doc = "3 - Pin Change Interrupt Request 0"]
    PCINT0 = 3,
    #[doc = "4 - Pin Change Interrupt Request 1"]
    PCINT1 = 4,
    #[doc = "5 - Pin Change Interrupt Request 2"]
    PCINT2 = 5,
    #[doc = "6 - Watchdog Time-out Interrupt"]
    WDT = 6,
    #[doc = "7 - Timer/Counter2 Compare Match A"]
    TIMER2_COMPA = 7,
    #[doc = "8 - Timer/Counter2 Compare Match B"]
    TIMER2_COMPB = 8,
    #[doc = "9 - Timer/Counter2 Overflow"]
    TIMER2_OVF = 9,
    #[doc = "10 - Timer/Counter1 Capture Event"]
    TIMER1_CAPT = 10,
    #[doc = "11 - Timer/Counter1 Compare Match A"]
    TIMER1_COMPA = 11,
    #[doc = "12 - Timer/Counter1 Compare Match B"]
    TIMER1_COMPB = 12,
    #[doc = "13 - Timer/Counter1 Overflow"]
    TIMER1_OVF = 13,
    #[doc = "14 - TimerCounter0 Compare Match A"]
    TIMER0_COMPA = 14,
    #[doc = "15 - TimerCounter0 Compare Match B"]
    TIMER0_COMPB = 15,
    #[doc = "16 - Timer/Couner0 Overflow"]
    TIMER0_OVF = 16,
    #[doc = "17 - SPI Serial Transfer Complete"]
    SPI0_STC = 17,
    #[doc = "18 - USART0 Rx Complete"]
    USART0_RX = 18,
    #[doc = "19 - USART0, Data Register Empty"]
    USART0_UDRE = 19,
    #[doc = "20 - USART0 Tx Complete"]
    USART0_TX = 20,
    #[doc = "21 - ADC Conversion Complete"]
    ADC = 21,
    #[doc = "22 - EEPROM Ready"]
    EE_READY = 22,
    #[doc = "23 - Analog Comparator"]
    ANALOG_COMP = 23,
    #[doc = "24 - Two-wire Serial Interface"]
    TWI0 = 24,
    #[doc = "25 - Store Program Memory Read"]
    SPM_READY = 25,
    #[doc = "26 - USART0 Start frame detection"]
    USART0_START = 26,
    #[doc = "27 - Pin Change Interrupt Request 3"]
    PCINT3 = 27,
    #[doc = "28 - USART1 Rx Complete"]
    USART1_RX = 28,
    #[doc = "29 - USART1, Data Register Empty"]
    USART1_UDRE = 29,
    #[doc = "30 - USART1 Tx Complete"]
    USART1_TX = 30,
    #[doc = "31 - USART1 Start frame detection"]
    USART1_START = 31,
    #[doc = "32 - Timer/Counter3 Capture Event"]
    TIMER3_CAPT = 32,
    #[doc = "33 - Timer/Counter3 Compare Match A"]
    TIMER3_COMPA = 33,
    #[doc = "34 - Timer/Counter3 Compare Match B"]
    TIMER3_COMPB = 34,
    #[doc = "35 - Timer/Counter3 Overflow"]
    TIMER3_OVF = 35,
    #[doc = "36 - Clock failure detection interrupt"]
    CFD = 36,
    #[doc = "37 - PTC End of conversion"]
    PTC_EOC = 37,
    #[doc = "38 - PTC Window comparator mode"]
    PTC_WCOMP = 38,
    #[doc = "39 - SPI1 Serial Transfer Complete"]
    SPI1_STC = 39,
    #[doc = "40 - TWI Transfer Complete"]
    TWI1 = 40,
    #[doc = "41 - Timer/Counter4 Capture Event"]
    TIMER4_CAPT = 41,
    #[doc = "42 - Timer/Counter4 Compare Match A"]
    TIMER4_COMPA = 42,
    #[doc = "43 - Timer/Counter4 Compare Match B"]
    TIMER4_COMPB = 43,
    #[doc = "44 - Timer/Counter4 Overflow"]
    TIMER4_OVF = 44,
}
#[derive(Debug, Copy, Clone)]
pub struct TryFromInterruptError(());
impl Interrupt {
    #[inline]
    pub fn try_from(value: u8) -> Result<Self, TryFromInterruptError> {
        match value {
            0 => Ok(Interrupt::RESET),
            1 => Ok(Interrupt::INT0),
            2 => Ok(Interrupt::INT1),
            3 => Ok(Interrupt::PCINT0),
            4 => Ok(Interrupt::PCINT1),
            5 => Ok(Interrupt::PCINT2),
            6 => Ok(Interrupt::WDT),
            7 => Ok(Interrupt::TIMER2_COMPA),
            8 => Ok(Interrupt::TIMER2_COMPB),
            9 => Ok(Interrupt::TIMER2_OVF),
            10 => Ok(Interrupt::TIMER1_CAPT),
            11 => Ok(Interrupt::TIMER1_COMPA),
            12 => Ok(Interrupt::TIMER1_COMPB),
            13 => Ok(Interrupt::TIMER1_OVF),
            14 => Ok(Interrupt::TIMER0_COMPA),
            15 => Ok(Interrupt::TIMER0_COMPB),
            16 => Ok(Interrupt::TIMER0_OVF),
            17 => Ok(Interrupt::SPI0_STC),
            18 => Ok(Interrupt::USART0_RX),
            19 => Ok(Interrupt::USART0_UDRE),
            20 => Ok(Interrupt::USART0_TX),
            21 => Ok(Interrupt::ADC),
            22 => Ok(Interrupt::EE_READY),
            23 => Ok(Interrupt::ANALOG_COMP),
            24 => Ok(Interrupt::TWI0),
            25 => Ok(Interrupt::SPM_READY),
            26 => Ok(Interrupt::USART0_START),
            27 => Ok(Interrupt::PCINT3),
            28 => Ok(Interrupt::USART1_RX),
            29 => Ok(Interrupt::USART1_UDRE),
            30 => Ok(Interrupt::USART1_TX),
            31 => Ok(Interrupt::USART1_START),
            32 => Ok(Interrupt::TIMER3_CAPT),
            33 => Ok(Interrupt::TIMER3_COMPA),
            34 => Ok(Interrupt::TIMER3_COMPB),
            35 => Ok(Interrupt::TIMER3_OVF),
            36 => Ok(Interrupt::CFD),
            37 => Ok(Interrupt::PTC_EOC),
            38 => Ok(Interrupt::PTC_WCOMP),
            39 => Ok(Interrupt::SPI1_STC),
            40 => Ok(Interrupt::TWI1),
            41 => Ok(Interrupt::TIMER4_CAPT),
            42 => Ok(Interrupt::TIMER4_COMPA),
            43 => Ok(Interrupt::TIMER4_COMPB),
            44 => Ok(Interrupt::TIMER4_OVF),
            _ => Err(TryFromInterruptError(())),
        }
    }
}
