mod macro_expr;
mod transform;

use proc_macro::TokenStream;

use proc_macro_error::{abort, emit_error, proc_macro_error};
use quote::{format_ident, quote};
use syn::{parse_macro_input, Item, Pat, ReturnType};

use transform::{Dependencies, Function, Scope, Var};

#[proc_macro_error]
#[proc_macro_attribute]
pub fn component(_metadata: TokenStream, input: TokenStream) -> TokenStream {
    let mut item = parse_macro_input!(input as Item);

    let item_fn = match item {
        Item::Fn(ref mut fun) => fun,
        _ => abort!(item, "component requires function input"),
    };

    let mut function = Function::new();
    let mut param_scope = Scope::new();

    if item_fn.sig.inputs.len() > 64 {
        emit_error!(
            item_fn.sig.inputs,
            "more than 64 properties are unsupported"
        );
    }

    let return_type = match &item_fn.sig.output {
        ReturnType::Type(_, ty) => ty,
        ReturnType::Default => {
            abort!(
                item_fn.sig,
                "input function must have return type avalanche::View"
            );
        }
    };

    if let Some(token) = &item_fn.sig.asyncness {
        abort!(token, "async components currently unsupported");
    };

    if let Some(token) = &item_fn.sig.unsafety {
        abort!(
            token,
            "unsafe components unsupported";
            note = "to write unsafe code, use an unsafe block"
        );
    };

    if let Some(token) = &item_fn.sig.variadic {
        abort!(token, "variadic components unsupported");
    };

    let inputs_len = item_fn.sig.inputs.len();

    let mut param_ident = Vec::with_capacity(inputs_len);
    let mut param_type = Vec::with_capacity(inputs_len);
    let mut param_attributes = Vec::with_capacity(inputs_len);
    let mut flag = Vec::with_capacity(inputs_len);

    // process render code
    for (i, param) in item_fn.sig.inputs.iter().enumerate() {
        let update_bit_pattern = 2u64.pow(i as u32);
        match param {
            syn::FnArg::Receiver(rec) => {
                abort!(rec, "receiver not allowed");
            }
            syn::FnArg::Typed(param) => {
                let ident = if let Pat::Ident(ident) = &*param.pat {
                    let mut dependencies = Dependencies::default();
                    dependencies.insert(ident.ident.to_owned());
                    param_scope.vars.push(Var {
                        name: ident.ident.to_string(),
                        dependencies: dependencies.into(),
                    });
                    ident
                } else {
                    abort!(param.pat, "expected identifier");
                };

                param_ident.push(ident);
                param_type.push(&param.ty);
                param_attributes.push(&ident.attrs);
                flag.push(update_bit_pattern);
            }
        }
    }

    // info for generating __last setter for last prop
    let last_param_ident = param_ident.last().into_iter();
    let last_param_type = param_type.last().into_iter();
    let last_param_attributes = param_attributes.last().into_iter();
    let last_flag = flag.last();

    function.scopes.push(param_scope);

    //dependencies unneeded: we only want to process the block
    let _ = function.block(&mut item_fn.block);

    let name = &item_fn.sig.ident;
    let builder_name = format_ident!("{}Builder", name);

    let render_body = &item_fn.block;
    let render_body_attributes = &item_fn.attrs;
    let visibility = &item_fn.vis;

    let component_default_impl = if inputs_len == 0 {
        Some(quote! {
            impl ::std::default::Default for #name {
                fn default() -> Self {
                    Self {
                        __internal_updates: 0,
                        __key: None,
                        __location: (0, 0)
                    }
                }
            }
        })
    } else {
        None
    };

    let component = quote! {
        #visibility struct #builder_name {
            __internal_updates: ::std::primitive::u64,
            __key: ::std::option::Option<::std::string::String>,
            #(#param_ident: ::std::option::Option<#param_type>),*
        }

        impl ::std::default::Default for #builder_name {
            fn default() -> Self {
                Self {
                    __internal_updates: 0,
                    __key: ::std::option::Option::None,
                    #(#param_ident: ::std::option::Option::None),*
                }
            }
        }

        impl #builder_name {
            fn new() -> Self {
                ::std::default::Default::default()
            }

            fn build(self, location: (::std::primitive::u32, ::std::primitive::u32)) -> #name {
                #name {
                    __internal_updates: self.__internal_updates,
                    __key: self.__key,
                    __location: location,
                    #(#param_ident: ::std::option::Option::unwrap(self.#param_ident)),*
                }
            }

            fn key<T: ::std::string::ToString>(mut self, key: T, _updated: ::std::primitive::bool) -> Self {
                //TODO: should updated be used?
                self.__key = ::std::option::Option::Some(::std::string::ToString::to_string(&key));
                self
            }

            #(
                #(#param_attributes)*
                fn #param_ident(mut self, val: #param_type, updated: ::std::primitive::bool) -> Self {
                    if updated {
                        self.__internal_updates |= #flag;
                    }
                    self.#param_ident = ::std::option::Option::Some(val);
                    self
                }
            )*

            #(
                #(#last_param_attributes)*
                fn __last(mut self, val: #last_param_type, updated: ::std::primitive::bool) -> Self {
                    if updated {
                        self.__internal_updates |= #last_flag;
                    }
                    self.#last_param_ident = ::std::option::Option::Some(val);
                    self
                }
            )*
        }

        #[derive(::std::clone::Clone)]
        #visibility struct #name {
            __internal_updates: ::std::primitive::u64,
            __key: std::option::Option<::std::string::String>,
            __location: (::std::primitive::u32, ::std::primitive::u32),
            #(#param_ident: #param_type),*
        }

        #component_default_impl

        impl avalanche::Component for #name where #( #param_type: ::std::clone::Clone ),* {
            type Builder = #builder_name;

            #( #render_body_attributes )*
            #[allow(clippy::eval_order_dependence, clippy::unit_arg)]
            fn render(&self, __avalanche_context: ::avalanche::Context) -> #return_type {
                // let state = ::std::option::Option::unwrap(::std::any::Any::downcast_mut::<#state_name>(&mut **__avalanche_context.state));

                let #name { #(#param_ident,)* .. } = ::std::clone::Clone::clone(self);

                #(let #param_ident = ::avalanche::Tracked::new(#param_ident, (&self.__internal_updates & #flag) != 0);)*

                let mut __avalanche_internal_updated = false;

                #render_body
            }

            fn updated(&self) -> ::std::primitive::bool {
                self.__internal_updates != 0
            }

            fn key(&self) -> ::std::option::Option<&str> {
                self.__key.as_deref()
            }

        }
    };

    // println!("{}", component);

    component.into()
}
