//! Error type for the authorization header library

use std::string::FromUtf8Error;

use http::header::InvalidHeaderValue;
use thiserror::Error;

/// Credentials decoding error
#[derive(Debug, Error)]
pub enum CredentialsDecodeError {
	/// Base64 header payload couldn't be decoded
	#[error(transparent)]
	Base64DecodeError(#[from] base64::DecodeError),
	/// decoded header isn't unicode
	#[error(transparent)]
	UnicodeDecodeError(#[from] FromUtf8Error),
	/// Password payload is missing a delimiter
	#[error("Delimiter missing, credentials must be in a user:pass format")]
	DelimiterNotFound,
}

#[derive(Debug, Error)]
/// Authorization header error
pub enum ParseError {
	/// Authorization header is missing from the header map
	#[error("Authorization header is missing from the header map")]
	AuthHeaderMissing,
	/// Authorization header contains invalid characters
	#[error("Authorization header contains invalid characters")]
	InvalidCharacters,
	/// Bad authorization header format
	#[error("Bad authorization header format")]
	BadFormat,
	/// The authentication type is unsupported
	#[error("The authentication type {0} is unsupported")]
	UnknownAuthenticationType(String),
	/// User credentials format is invalid
	#[error("User credentials format is invalid")]
	InvalidUserCredentialsFormat,
	/// Credentials can't be decoded
	#[error(transparent)]
	CredentialsDecodeError(#[from] CredentialsDecodeError),
	/// Invalid header value
	#[error(transparent)]
	InvalidHeaderValue(#[from] InvalidHeaderValue),
}

impl ParseError {
	/// The authentication type is unsupported
	pub fn unknown_authentication_type<S: Into<String>>(auth_type: S) -> Self {
		Self::UnknownAuthenticationType(auth_type.into())
	}
}
