/*
 * File: twincat_link.rs
 * Project: src
 * Created Date: 27/05/2021
 * Author: Shun Suzuki
 * -----
 * Last Modified: 27/05/2021
 * Modified By: Shun Suzuki (suzuki@hapis.k.u-tokyo.ac.jp)
 * -----
 * Copyright (c) 2021 Hapis Lab. All rights reserved.
 *
 */

use anyhow::Result;
use libc::c_void;

use autd3_core::link::Link;

use crate::{error::AdsError, native_methods::*};

const INDEX_GROUP: u32 = 0x0304_0030;
const INDEX_OFFSET_BASE: u32 = 0x8100_0000;
const INDEX_OFFSET_BASE_READ: u32 = 0x8000_0000;
const PORT: u16 = 301;

pub struct TwinCatLink {
    port: i32,
    send_addr: AmsAddr,
}

impl TwinCatLink {
    pub fn new() -> Self {
        unsafe {
            let ams_addr: AmsAddr = std::mem::zeroed();
            Self {
                port: 0,
                send_addr: AmsAddr {
                    net_id: ams_addr.net_id,
                    port: PORT,
                },
            }
        }
    }
}

impl Default for TwinCatLink {
    fn default() -> Self {
        Self::new()
    }
}

impl Link for TwinCatLink {
    fn open(&mut self) -> Result<()> {
        unsafe {
            let port = (TC_ADS.tc_ads_port_open)();
            if port == 0 {
                return Err(AdsError::FailedOpenPort.into());
            }
            self.port = port;

            let mut ams_addr: AmsAddr = std::mem::zeroed();
            let n_err = (TC_ADS.tc_ads_get_local_address)(port, &mut ams_addr as *mut _);
            if n_err != 0 {
                return Err(AdsError::FailedGetLocalAddress(n_err).into());
            }
            self.send_addr.net_id = ams_addr.net_id;
        }

        Ok(())
    }

    fn close(&mut self) -> Result<()> {
        unsafe {
            (TC_ADS.tc_ads_port_close)(self.port);
        }
        self.port = 0;
        Ok(())
    }

    fn send(&mut self, data: &[u8]) -> Result<bool> {
        unsafe {
            let n_err = (TC_ADS.tc_ads_sync_write_req)(
                self.port,
                &self.send_addr as *const _,
                INDEX_GROUP,
                INDEX_OFFSET_BASE,
                data.len() as u32,
                data.as_ptr() as *const c_void,
            );

            if n_err > 0 {
                Err(AdsError::FailedSendData(n_err).into())
            } else {
                Ok(true)
            }
        }
    }

    fn read(&mut self, data: &mut [u8]) -> Result<bool> {
        let mut read_bytes: u32 = 0;
        unsafe {
            let n_err = (TC_ADS.tc_ads_sync_read_req)(
                self.port,
                &self.send_addr as *const _,
                INDEX_GROUP,
                INDEX_OFFSET_BASE_READ,
                data.len() as u32,
                data.as_mut_ptr() as *mut c_void,
                &mut read_bytes as *mut u32,
            );

            if n_err > 0 {
                Err(AdsError::FailedReadData(n_err).into())
            } else {
                Ok(true)
            }
        }
    }

    fn is_open(&self) -> bool {
        self.port > 0
    }
}
