/*
 * File: link_soem.rs
 * Project: src
 * Created Date: 27/04/2022
 * Author: Shun Suzuki
 * -----
 * Last Modified: 04/05/2022
 * Modified By: Shun Suzuki (suzuki@hapis.k.u-tokyo.ac.jp)
 * -----
 * Copyright (c) 2022 Hapis Lab. All rights reserved.
 *
 */

use std::{
    sync::{
        atomic::{AtomicBool, Ordering},
        Arc, Mutex,
    },
    thread::{self, JoinHandle},
    time::Duration,
    usize,
};

use anyhow::Result;
use crossbeam_channel::{bounded, Sender};
use libc::c_void;

use autd3_core::{
    error::AUTDInternalError, link::Link, RxDatagram, TxDatagram, EC_SYNC0_CYCLE_TIME_NANO_SEC,
};

use crate::{
    ecat_thread::{EcatErrorHandler, EcatThreadHandler, HighPrecision, Normal},
    error::SOEMError,
    iomap::IOMap,
    native_methods::*,
    Config,
};

const SEND_BUF_SIZE: usize = 32;

pub struct SOEM<F: Fn(&str) + Send> {
    ecatth_handle: Option<JoinHandle<()>>,
    error_handle: Option<F>,
    is_open: bool,
    ifname: std::ffi::CString,
    config: Config,
    dev_num: u16,
    sender: Option<Sender<TxDatagram>>,
    recv_thread: Option<JoinHandle<()>>,
    thread_running: Arc<AtomicBool>,
    rx: Arc<Mutex<RxDatagram>>,
    ec_sync0_cycle_time_ns: u32,
}

impl<F: Fn(&str) + Send> SOEM<F> {
    pub fn new(ifname: &str, dev_num: u16, config: Config, error_handle: F) -> Self {
        let ec_sync0_cycle_time_ns = EC_SYNC0_CYCLE_TIME_NANO_SEC * config.cycle_ticks as u32;
        Self {
            dev_num,
            ecatth_handle: None,
            error_handle: Some(error_handle),
            is_open: false,
            ifname: std::ffi::CString::new(ifname.to_string()).unwrap(),
            sender: None,
            rx: Arc::new(Mutex::new(RxDatagram::new(dev_num as _))),
            recv_thread: None,
            thread_running: Arc::new(AtomicBool::new(false)),
            config,
            ec_sync0_cycle_time_ns,
        }
    }
}

unsafe extern "C" fn dc_config(context: *mut ecx_contextt, slave: u16) -> i32 {
    let cyc_time = *((*context).userdata as *mut u32);
    ec_dcsync0(slave, 1, cyc_time, 0);
    0
}

impl<F: 'static + Fn(&str) + Send> Link for SOEM<F> {
    fn open(&mut self) -> Result<()> {
        let mut io_map = Box::new(IOMap::new(self.dev_num as _));

        let (tx_sender, tx_receiver) = bounded(SEND_BUF_SIZE);
        let (rx_sender, rx_receiver) = bounded(SEND_BUF_SIZE);

        let rx = self.rx.clone();
        let thread_running = self.thread_running.clone();
        thread_running.store(true, Ordering::Release);
        self.recv_thread = Some(thread::spawn(move || {
            while thread_running.load(Ordering::Acquire) {
                if let Ok(data) = rx_receiver.recv_timeout(Duration::from_millis(100)) {
                    rx.lock().unwrap().copy_from(&data);
                }
            }
        }));

        unsafe {
            if ec_init(self.ifname.as_ptr()) <= 0 {
                return Err(SOEMError::NoSocketConnection(
                    self.ifname.to_str().unwrap().to_string(),
                )
                .into());
            }

            let wc = ec_config_init(0);
            if wc <= 0 {
                return Err(SOEMError::SlaveNotFound(0, self.dev_num).into());
            }
            if wc as u16 != self.dev_num {
                return Err(SOEMError::SlaveNotFound(wc as u16, self.dev_num).into());
            }

            ecx_context.userdata = &mut self.ec_sync0_cycle_time_ns as *mut _ as *mut c_void;

            (1..=ec_slavecount as usize).for_each(|i| {
                ec_slave[i].PO2SOconfigx = Some(dc_config);
            });

            ec_configdc();

            ec_config_map(io_map.data() as *mut c_void);

            ec_statecheck(
                0,
                ec_state_EC_STATE_SAFE_OP as u16,
                EC_TIMEOUTSTATE as i32 * 4,
            );

            ec_readstate();

            ec_slave[0].state = ec_state_EC_STATE_OPERATIONAL as u16;

            ec_writestate(0);

            let expected_wkc = (ec_group[0].outputsWKC * 2 + ec_group[0].inputsWKC) as i32;
            let cycletime = self.ec_sync0_cycle_time_ns as i64;
            let error_handle = self.error_handle.take();
            let thread_running = self.thread_running.clone();
            let is_high_precision = self.config.high_precision_timer;
            self.ecatth_handle = Some(std::thread::spawn(move || {
                let error_handler = EcatErrorHandler { error_handle };
                if is_high_precision {
                    let mut callback = EcatThreadHandler::<_, HighPrecision>::new(
                        io_map,
                        thread_running,
                        tx_receiver,
                        rx_sender,
                        expected_wkc,
                        cycletime,
                        error_handler,
                    );
                    callback.run();
                } else {
                    let mut callback = EcatThreadHandler::<_, Normal>::new(
                        io_map,
                        thread_running,
                        tx_receiver,
                        rx_sender,
                        expected_wkc,
                        cycletime,
                        error_handler,
                    );
                    callback.run();
                }
            }));

            std::thread::sleep(std::time::Duration::from_millis(100));

            ec_statecheck(
                0,
                ec_state_EC_STATE_OPERATIONAL as u16,
                EC_TIMEOUTSTATE as i32 * 5,
            );

            if ec_slave[0].state != ec_state_EC_STATE_OPERATIONAL as u16 {
                return Err(SOEMError::NotResponding.into());
            }
        }

        self.is_open = true;
        self.sender = Some(tx_sender);

        Ok(())
    }

    fn close(&mut self) -> Result<()> {
        if !self.is_open {
            return Ok(());
        }

        while !self.sender.as_ref().unwrap().is_empty() {
            std::thread::sleep(std::time::Duration::from_nanos(
                self.ec_sync0_cycle_time_ns as _,
            ));
        }

        self.is_open = false;

        std::thread::sleep(std::time::Duration::from_millis(200));

        self.thread_running.store(false, Ordering::Release);
        if let Some(timer) = self.ecatth_handle.take() {
            let _ = timer.join();
        }

        if let Some(th) = self.recv_thread.take() {
            let _ = th.join();
        }

        unsafe {
            let cyc_time = *(ecx_context.userdata as *mut u32);
            (1..=ec_slavecount as u16).for_each(|i| {
                ec_dcsync0(i, 0, cyc_time, 0);
            });

            ec_slave[0].state = ec_state_EC_STATE_SAFE_OP as _;
            ec_writestate(0);
            ec_statecheck(0, ec_state_EC_STATE_SAFE_OP as _, EC_TIMEOUTSTATE as _);

            ec_slave[0].state = ec_state_EC_STATE_PRE_OP as _;
            ec_writestate(0);
            ec_statecheck(0, ec_state_EC_STATE_PRE_OP as _, EC_TIMEOUTSTATE as _);

            ec_close();
        }

        Ok(())
    }

    fn send(&mut self, tx: &TxDatagram) -> Result<bool> {
        let buf = tx.clone();

        self.sender.as_mut().unwrap().send(buf)?;

        Ok(true)
    }

    fn receive(&mut self, rx: &mut RxDatagram) -> Result<bool> {
        if !self.is_open {
            return Err(AUTDInternalError::LinkClosed.into());
        }

        rx.copy_from(&self.rx.lock().unwrap());

        Ok(true)
    }

    fn cycle_ticks(&self) -> u16 {
        self.config.cycle_ticks
    }

    fn is_open(&self) -> bool {
        self.is_open
    }
}
