use crate as aurum_actors;
use crate::core::{Actor, ActorContext, Case, UnifiedType};
use crate::AurumInterface;
use async_trait::async_trait;
use std::fmt::Display;
use LogLevel::*;
use LoggerMsg::*;

#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub enum LogLevel {
  Trace,
  Debug,
  Info,
  Warn,
  Error,
  Fatal,
  Off,
}
impl LogLevel {
  pub const MIN: LogLevel = LogLevel::Trace;

  pub const fn caps(&self) -> &'static str {
    match self {
      Trace => "TRACE",
      Debug => "DEBUG",
      Info => "INFO",
      Warn => "WARN",
      Error => "ERROR",
      Fatal => "FATAL",
      Off => "OFF",
    }
  }
}

pub enum LogSpecial {
  SentBytes(u64),
  RecvdBytes(u64),
}

#[derive(AurumInterface)]
#[aurum(local)]
pub enum LoggerMsg {
  Log {
    level: LogLevel,
    file: &'static str,
    line: u32,
    column: u32,
    msg: Box<dyn Display + Send + 'static>,
  },
  SetLevel(LogLevel),
  Special(LogSpecial),
}

pub struct Logger {
  bytes_sent: u64,
  bytes_recvd: u64,
  level: LogLevel,
}
impl Logger {
  pub fn new(level: LogLevel) -> Self {
    Logger {
      bytes_sent: 0,
      bytes_recvd: 0,
      level: level,
    }
  }
}
#[async_trait]
impl<U: Case<LoggerMsg> + UnifiedType> Actor<U, LoggerMsg> for Logger {
  async fn recv(&mut self, ctx: &ActorContext<U, LoggerMsg>, msg: LoggerMsg) {
    match msg {
      Log {
        level,
        file,
        line,
        column,
        msg: log,
      } => {
        if level >= self.level {
          println!(
            "{} {}:{}:{} - {} - {}",
            level.caps(),
            file,
            line,
            column,
            ctx.node.socket(),
            log
          );
        }
      }
      SetLevel(level) => self.level = level,
      Special(s) => match s {
        LogSpecial::SentBytes(b) => self.bytes_sent += b,
        LogSpecial::RecvdBytes(b) => self.bytes_recvd += b,
      },
    }
  }
}

#[macro_export]
macro_rules! log {
  ($env_level:expr, $node:expr, $msg_level:expr, $msg:expr) => {
    if $msg_level >= $env_level {
      $node.log($crate::testkit::LoggerMsg::Log {
        level: $msg_level,
        file: ::std::file!(),
        line: ::std::line!(),
        column: ::std::column!(),
        msg: ::std::boxed::Box::new($msg),
      });
    }
  };
}

#[macro_export]
macro_rules! trace {
  ($env_level:expr, $node:expr, $msg:expr) => {
    $crate::log!($env_level, $node, $crate::testkit::LogLevel::Trace, $msg);
  };
}

#[macro_export]
macro_rules! debug {
  ($env_level:expr, $node:expr, $msg:expr) => {
    $crate::log!($env_level, $node, $crate::testkit::LogLevel::Debug, $msg);
  };
}

#[macro_export]
macro_rules! info {
  ($env_level:expr, $node:expr, $msg:expr) => {
    $crate::log!($env_level, $node, $crate::testkit::LogLevel::Info, $msg);
  };
}

#[macro_export]
macro_rules! warn {
  ($env_level:expr, $node:expr, $msg:expr) => {
    $crate::log!($env_level, $node, $crate::testkit::LogLevel::Warn, $msg);
  };
}

#[macro_export]
macro_rules! error {
  ($env_level:expr, $node:expr, $msg:expr) => {
    $crate::log!($env_level, $node, $crate::testkit::LogLevel::Error, $msg);
  };
}

#[macro_export]
macro_rules! fatal {
  ($env_level:expr, $node:expr, $msg:expr) => {
    $crate::log!($env_level, $node, $crate::testkit::LogLevel::Fatal, $msg);
  };
}
