use std::collections::HashMap;

use async_trait::async_trait;
use raur::{Error, Package, Raur, SearchBy};

pub struct MockRaur {
    pub pkgs: HashMap<String, Package>,
}

#[async_trait]
impl Raur for MockRaur {
    async fn search_by<S: AsRef<str> + Send + Sync>(
        &self,
        query: S,
        _by: SearchBy,
    ) -> Result<Vec<Package>, Error> {
        Ok(self
            .pkgs
            .values()
            .filter(|p| {
                p.name.contains(query.as_ref())
                    || p.description
                        .as_ref()
                        .map(|s| s.as_str())
                        .unwrap_or_else(|| "")
                        .contains(query.as_ref())
            })
            .cloned()
            .collect())
    }

    async fn raw_info<S: AsRef<str> + Send + Sync>(
        &self,
        pkg_names: &[S],
    ) -> Result<Vec<Package>, Error> {
        let mut out = Vec::new();

        for name in pkg_names {
            if let Some(pkg) = self.pkgs.get(name.as_ref()) {
                out.push(pkg.clone());
            }
        }

        Ok(out)
    }
}

impl MockRaur {
    pub fn new() -> Self {
        MockRaur {
            pkgs: HashMap::new(),
        }
    }

    pub fn pkg<S: Into<String>>(&mut self, name: S) -> MockPackage {
        let name = name.into();
        let pkg = Package {
            id: 0,
            name: name.clone(),
            package_base_id: 0,
            package_base: "name".into(),
            version: "0".into(),
            description: None,
            url: None,
            num_votes: 0,
            popularity: 0.0,
            out_of_date: None,
            maintainer: None,
            first_submitted: 0,
            last_modified: 0,
            url_path: "".into(),
            groups: vec![],
            depends: vec![],
            make_depends: vec![],
            opt_depends: vec![],
            check_depends: vec![],
            conflicts: vec![],
            replaces: vec![],
            provides: vec![],
            license: vec![],
            keywords: vec![],
        };

        self.pkgs.insert(pkg.name.clone(), pkg);
        MockPackage(self.pkgs.get_mut(&name).unwrap())
    }
}

pub struct MockPackage<'a>(&'a mut Package);

impl<'a> MockPackage<'a> {
    pub fn depend<S: Into<String>>(self, s: S) -> Self {
        self.0.depends.push(s.into());
        self
    }

    pub fn make_depend<S: Into<String>>(self, s: S) -> Self {
        self.0.make_depends.push(s.into());
        self
    }

    pub fn provide<S: Into<String>>(self, s: S) -> Self {
        self.0.provides.push(s.into());
        self
    }

    pub fn conflict<S: Into<String>>(self, s: S) -> Self {
        self.0.conflicts.push(s.into());
        self
    }

    pub fn version<S: Into<String>>(self, s: S) -> Self {
        self.0.version = s.into();
        self
    }
}

pub fn raur() -> impl Raur {
    let mut raur = MockRaur::new();
    raur.pkg("a").depend("b>1");
    raur.pkg("b").version("1");

    raur.pkg("flann")
        .depend("lz4")
        .depend("hdf5")
        .make_depend("cmake")
        .make_depend("python2")
        .make_depend("texlive-core");
    raur.pkg("gazebo")
        .depend("boost>=1.40.0")
        .depend("curl>=4.0")
        .depend("freeglut")
        .depend("freeimage>=3.0")
        .depend("intel-tbb>=3.0")
        .depend("libccd>=1.4")
        .depend("libltdl>=2.4.2")
        .depend("libtar>=1.2")
        .depend("libxml2>=2.7.7")
        .depend("ogre-1.9")
        .depend("protobuf>=2.3.0")
        .depend("sdformat=6")
        .depend("ignition-math=4")
        .depend("ignition-transport=4")
        .depend("ignition-common")
        .depend("ignition-fuel_tools")
        .depend("ignition-msgs")
        .depend("tinyxml2")
        .depend("qwt")
        .make_depend("cmake")
        .make_depend("doxygen")
        .make_depend("ignition-cmake");
    raur.pkg("ignition-common")
        .depend("ignition-math>=6")
        .depend("tinyxml2")
        .depend("freeimage")
        .depend("libutil-linux")
        .depend("gts")
        .depend("ffmpeg")
        .make_depend("ignition-cmake>=2")
        .make_depend("util-linux");
    raur.pkg("ignition-cmake")
        .depend("cmake")
        .depend("pkgconfig")
        .depend("ruby-ronn")
        .depend("doxygen");
    raur.pkg("ignition-fuel_tools")
        .depend("curl")
        .depend("jsoncpp")
        .depend("libyaml")
        .depend("libzip")
        .depend("ignition-common>=3")
        .make_depend("ignition-cmake>=2");
    raur.pkg("libccd").provide("libccd").make_depend("cmake");
    raur.pkg("opencv3-opt")
        .provide("opencv3")
        .depend("intel-tbb")
        .depend("openexr")
        .depend("gst-plugins-base")
        .depend("libdc1394")
        .depend("cblas")
        .depend("lapack")
        .depend("libgphoto2")
        .depend("jasper")
        .depend("ffmpeg")
        .make_depend("cmake")
        .make_depend("python-numpy")
        .make_depend("python-setuptools")
        .make_depend("mesa")
        .make_depend("eigen")
        .make_depend("hdf5")
        .make_depend("lapacke")
        .make_depend("gtk3")
        .make_depend("vtk")
        .make_depend("glew")
        .make_depend("double-conversion");
    raur.pkg("pcl")
        .depend("boost")
        .depend("eigen")
        .depend("flann")
        .depend("vtk")
        .depend("qhull")
        .depend("qt5-base")
        .depend("glu")
        .depend("qt5-webkit")
        .depend("openmpi")
        .depend("python2")
        .depend("libxt")
        .depend("libharu")
        .depend("proj")
        .depend("glew")
        .depend("netcdf")
        .depend("libusb")
        .make_depend("cmake")
        .make_depend("gl2ps")
        .make_depend("python");
    raur.pkg("python-catkin_pkg")
        .provide("python-catkin-pkg")
        .depend("python")
        .depend("python-argparse")
        .depend("python-dateutil")
        .depend("python-docutils")
        .make_depend("python-setuptools");
    raur.pkg("python-empy").depend("python");
    raur.pkg("python-rosdep")
        .depend("python")
        .depend("python-catkin_pkg")
        .depend("python-rosdistro")
        .depend("python-rospkg")
        .depend("python-yaml")
        .make_depend("python-setuptools");
    raur.pkg("python-rosdistro")
        .depend("python")
        .depend("python-catkin_pkg")
        .depend("python-rospkg")
        .depend("python-yaml")
        .make_depend("python-setuptools");
    raur.pkg("python-rospkg")
        .depend("python")
        .depend("python-yaml")
        .make_depend("python-setuptools");
    raur.pkg("ros-build-tools");
    raur.pkg("ros-build-tools-py3")
        .provide("ros-build-tools")
        .depend("bash");
    raur.pkg("ros-melodic-camera-calibration")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-image-geometry")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-camera-calibration-parsers")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-roscpp-serialization")
        .depend("boost")
        .depend("yaml-cpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-roscpp-serialization")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("yaml-cpp")
        .make_depend("boost")
        .make_depend("pkg-config");
    raur.pkg("ros-melodic-camera-info-manager")
        .depend("ros-melodic-camera-calibration-parsers")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-image-transport")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-roslib")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-camera-calibration-parsers")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-image-transport")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-roslib")
        .make_depend("boost");
    raur.pkg("ros-melodic-catkin")
        .depend("python-nose")
        .depend("gtest")
        .depend("python-catkin_pkg")
        .depend("python-empy")
        .depend("gmock")
        .depend("python")
        .depend("ros-build-tools-py3")
        .make_depend("cmake")
        .make_depend("python-catkin_pkg")
        .make_depend("python-empy")
        .make_depend("python");
    raur.pkg("ros-melodic-cmake-modules")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-compressed-depth-image-transport")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-image-transport");
    raur.pkg("ros-melodic-compressed-image-transport")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-image-transport");
    raur.pkg("ros-melodic-control-msgs")
        .depend("ros-melodic-trajectory-msgs")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-trajectory-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-actionlib-msgs")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-control-toolbox")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-control-msgs")
        .depend("ros-melodic-realtime-tools")
        .depend("ros-melodic-cmake-modules")
        .depend("tinyxml")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-control-msgs")
        .make_depend("ros-melodic-realtime-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("tinyxml");
    raur.pkg("ros-melodic-controller-interface")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-hardware-interface")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-hardware-interface")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-controller-manager")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-controller-manager-msgs")
        .depend("ros-melodic-hardware-interface")
        .depend("ros-melodic-controller-interface")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-controller-manager-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-hardware-interface")
        .make_depend("ros-melodic-controller-interface");
    raur.pkg("ros-melodic-controller-manager-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-std-msgs");
    raur.pkg("ros-melodic-cv-bridge")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-rosconsole")
        .depend("boost")
        .depend("python")
        .depend("python-numpy")
        .depend("opencv3-opt")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-catkin")
        .make_depend("boost")
        .make_depend("python")
        .make_depend("python-numpy")
        .make_depend("opencv3-opt");
    raur.pkg("ros-melodic-depth-image-proc")
        .depend("ros-melodic-image-geometry")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-eigen-conversions")
        .depend("ros-melodic-tf2-ros")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-image-transport")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf2")
        .make_depend("ros-melodic-image-geometry")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-stereo-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-eigen-conversions")
        .make_depend("ros-melodic-tf2-ros")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-image-transport")
        .make_depend("boost");
    raur.pkg("ros-melodic-desktop")
        .depend("ros-melodic-angles")
        .depend("ros-melodic-common-tutorials")
        .depend("ros-melodic-urdf-tutorial")
        .depend("ros-melodic-geometry-tutorials")
        .depend("ros-melodic-visualization-tutorials")
        .depend("ros-melodic-viz")
        .depend("ros-melodic-roslint")
        .depend("ros-melodic-robot")
        .depend("ros-melodic-ros-tutorials")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-desktop-full")
        .depend("ros-melodic-desktop")
        .depend("ros-melodic-simulators")
        .depend("ros-melodic-perception")
        .depend("ros-melodic-urdf-sim-tutorial")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-diagnostic-aggregator")
        .depend("ros-melodic-bondcpp")
        .depend("ros-melodic-xmlrpcpp")
        .depend("ros-melodic-diagnostic-msgs")
        .depend("ros-melodic-bondpy")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-pluginlib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-bondcpp")
        .make_depend("ros-melodic-xmlrpcpp")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-diagnostic-msgs")
        .make_depend("ros-melodic-bondpy")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-pluginlib");
    raur.pkg("ros-melodic-diagnostic-analysis")
        .depend("ros-melodic-roslib")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-diagnostic-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-diagnostic-msgs")
        .make_depend("ros-melodic-rosbag")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-roslib");
    raur.pkg("ros-melodic-diagnostic-common-diagnostics")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-diagnostic-updater")
        .depend("python-psutil")
        .depend("hddtemp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-diagnostic-updater");
    raur.pkg("ros-melodic-diagnostic-updater")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-diagnostic-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-diagnostic-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-roscpp");
    raur.pkg("ros-melodic-diagnostics")
        .depend("ros-melodic-diagnostic-updater")
        .depend("ros-melodic-diagnostic-analysis")
        .depend("ros-melodic-diagnostic-common-diagnostics")
        .depend("ros-melodic-diagnostic-aggregator")
        .depend("ros-melodic-self-test")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-diff-drive-controller")
        .depend("ros-melodic-urdf")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-realtime-tools")
        .depend("ros-melodic-nav-msgs")
        .depend("ros-melodic-controller-interface")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-controller-manager")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-urdf")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-realtime-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-nav-msgs")
        .make_depend("ros-melodic-controller-interface")
        .make_depend("ros-melodic-tf");
    raur.pkg("ros-melodic-eigen-conversions")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-geometry-msgs")
        .depend("eigen3")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("eigen3");
    raur.pkg("ros-melodic-executive-smach")
        .depend("ros-melodic-smach-msgs")
        .depend("ros-melodic-smach")
        .depend("ros-melodic-smach-ros")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-filters")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-roslib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-roslib");
    raur.pkg("ros-melodic-forward-command-controller")
        .depend("ros-melodic-controller-interface")
        .depend("ros-melodic-realtime-tools")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-hardware-interface")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-realtime-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-hardware-interface")
        .make_depend("ros-melodic-controller-interface")
        .make_depend("ros-melodic-std-msgs");
    raur.pkg("ros-melodic-gazebo-dev")
        .depend("gazebo")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-gazebo-msgs")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-trajectory-msgs")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-geometry-msgs")
        .depend("ros-melodic-message-runtime")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-trajectory-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-std-srvs")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-sensor-msgs");
    raur.pkg("ros-melodic-gazebo-plugins")
        .depend("ros-melodic-diagnostic-updater")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-gazebo-msgs")
        .depend("ros-melodic-geometry-msgs")
        .depend("ros-melodic-trajectory-msgs")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-tf2-ros")
        .depend("ros-melodic-urdf")
        .depend("ros-melodic-nav-msgs")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-camera-info-manager")
        .depend("ros-melodic-angles")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-polled-camera")
        .depend("ros-melodic-image-transport")
        .depend("ros-melodic-gazebo-dev")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-diagnostic-updater")
        .make_depend("ros-melodic-rosgraph-msgs")
        .make_depend("ros-melodic-std-srvs")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-gazebo-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-gazebo-dev")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-trajectory-msgs")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-tf2-ros")
        .make_depend("ros-melodic-urdf")
        .make_depend("ros-melodic-nav-msgs")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-camera-info-manager")
        .make_depend("ros-melodic-angles")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-polled-camera")
        .make_depend("ros-melodic-image-transport")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-rospy");
    raur.pkg("ros-melodic-gazebo-ros")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-gazebo-msgs")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-roslib")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-geometry-msgs")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-gazebo-dev")
        .depend("tinyxml")
        .depend("python")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-gazebo-msgs")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-roslib")
        .make_depend("ros-melodic-std-srvs")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("ros-melodic-rosgraph-msgs")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-gazebo-dev")
        .make_depend("tinyxml");
    raur.pkg("ros-melodic-gazebo-ros-control")
        .depend("ros-melodic-joint-limits-interface")
        .depend("ros-melodic-controller-manager")
        .depend("ros-melodic-hardware-interface")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-angles")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-urdf")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-gazebo-ros")
        .depend("ros-melodic-control-toolbox")
        .depend("ros-melodic-transmission-interface")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-joint-limits-interface")
        .make_depend("ros-melodic-controller-manager")
        .make_depend("ros-melodic-transmission-interface")
        .make_depend("ros-melodic-hardware-interface")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-angles")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-urdf")
        .make_depend("ros-melodic-control-toolbox")
        .make_depend("ros-melodic-gazebo-dev");
    raur.pkg("ros-melodic-gazebo-ros-pkgs")
        .depend("ros-melodic-gazebo-msgs")
        .depend("ros-melodic-gazebo-ros")
        .depend("ros-melodic-gazebo-plugins")
        .depend("ros-melodic-gazebo-dev")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-geometry")
        .depend("ros-melodic-angles")
        .depend("ros-melodic-kdl-conversions")
        .depend("ros-melodic-tf-conversions")
        .depend("ros-melodic-eigen-conversions")
        .depend("ros-melodic-tf")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-hardware-interface")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-image-common")
        .depend("ros-melodic-polled-camera")
        .depend("ros-melodic-camera-calibration-parsers")
        .depend("ros-melodic-camera-info-manager")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-image-geometry")
        .depend("ros-melodic-sensor-msgs")
        .depend("opencv")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("opencv");
    raur.pkg("ros-melodic-image-pipeline")
        .depend("ros-melodic-image-rotate")
        .depend("ros-melodic-stereo-image-proc")
        .depend("ros-melodic-depth-image-proc")
        .depend("ros-melodic-image-view")
        .depend("ros-melodic-image-proc")
        .depend("ros-melodic-image-publisher")
        .depend("ros-melodic-camera-calibration")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-image-proc")
        .depend("ros-melodic-image-geometry")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-nodelet-topic-tools")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-image-geometry")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-nodelet-topic-tools")
        .make_depend("ros-melodic-image-transport")
        .make_depend("boost");
    raur.pkg("ros-melodic-image-publisher")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-camera-info-manager")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-camera-info-manager")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-image-transport");
    raur.pkg("ros-melodic-image-rotate")
        .depend("ros-melodic-tf2-geometry-msgs")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-tf2-ros")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf2-geometry-msgs")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-tf2-ros")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-tf2")
        .make_depend("ros-melodic-image-transport");
    raur.pkg("ros-melodic-image-transport-plugins")
        .depend("ros-melodic-compressed-depth-image-transport")
        .depend("ros-melodic-compressed-image-transport")
        .depend("ros-melodic-theora-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-image-view")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-camera-calibration-parsers")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-image-transport")
        .depend("gtk2")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-std-srvs")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-stereo-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-camera-calibration-parsers")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-image-transport")
        .make_depend("gtk2");
    raur.pkg("ros-melodic-joint-limits-interface")
        .depend("ros-melodic-urdf")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-hardware-interface")
        .depend("urdfdom")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-urdf")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-hardware-interface")
        .make_depend("ros-melodic-catkin")
        .make_depend("urdfdom");
    raur.pkg("ros-melodic-joint-state-controller")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-realtime-tools")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-hardware-interface")
        .depend("ros-melodic-controller-interface")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-realtime-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-hardware-interface")
        .make_depend("ros-melodic-controller-interface");
    raur.pkg("ros-melodic-kdl-conversions")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-kdl-parser-py")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-urdf")
        .depend("ros-melodic-python-orocos-kdl")
        .depend("ros-melodic-urdfdom-py")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("ros-melodic-urdf")
        .make_depend("ros-melodic-catkin")
        .make_depend("python-catkin_pkg");
    raur.pkg("ros-melodic-laser-assembler")
        .depend("ros-melodic-filters")
        .depend("ros-melodic-laser-geometry")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-pluginlib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-filters")
        .make_depend("ros-melodic-laser-geometry")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-pluginlib");
    raur.pkg("ros-melodic-laser-filters")
        .depend("ros-melodic-filters")
        .depend("ros-melodic-laser-geometry")
        .depend("ros-melodic-angles")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-pluginlib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-filters")
        .make_depend("ros-melodic-laser-geometry")
        .make_depend("ros-melodic-angles")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-pluginlib");
    raur.pkg("ros-melodic-laser-pipeline")
        .depend("ros-melodic-laser-geometry")
        .depend("ros-melodic-laser-assembler")
        .depend("ros-melodic-laser-filters")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-pcl-conversions")
        .depend("ros-melodic-pcl-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-std-msgs")
        .depend("eigen3")
        .depend("pcl")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-pcl-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-std-msgs");
    raur.pkg("ros-melodic-pcl-ros")
        .depend("ros-melodic-pcl-conversions")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-tf2-eigen")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-nodelet-topic-tools")
        .depend("ros-melodic-pcl-msgs")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-tf")
        .depend("eigen3")
        .depend("pcl")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-pcl-conversions")
        .make_depend("ros-melodic-nodelet-topic-tools")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-tf2-eigen")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-roslib")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-rosbag")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-pcl-msgs")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("eigen3")
        .make_depend("pcl");
    raur.pkg("ros-melodic-perception")
        .depend("ros-melodic-vision-opencv")
        .depend("ros-melodic-ros-base")
        .depend("ros-melodic-perception-pcl")
        .depend("ros-melodic-laser-pipeline")
        .depend("ros-melodic-image-transport-plugins")
        .depend("ros-melodic-image-pipeline")
        .depend("ros-melodic-image-common")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-perception-pcl")
        .depend("ros-melodic-pcl-conversions")
        .depend("ros-melodic-pcl-ros")
        .depend("ros-melodic-pcl-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-polled-camera")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-image-transport");
    raur.pkg("ros-melodic-position-controllers")
        .depend("ros-melodic-controller-interface")
        .depend("ros-melodic-forward-command-controller")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-controller-interface")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-forward-command-controller");
    raur.pkg("ros-melodic-python-orocos-kdl")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-catkin")
        .depend("python-sip")
        .depend("sip")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("python-sip")
        .make_depend("sip");
    raur.pkg("ros-melodic-realtime-tools")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rospy");
    raur.pkg("ros-melodic-robot")
        .depend("ros-melodic-filters")
        .depend("ros-melodic-ros-base")
        .depend("ros-melodic-joint-state-publisher")
        .depend("ros-melodic-executive-smach")
        .depend("ros-melodic-urdf-parser-plugin")
        .depend("ros-melodic-xacro")
        .depend("ros-melodic-diagnostics")
        .depend("ros-melodic-robot-state-publisher")
        .depend("ros-melodic-kdl-parser-py")
        .depend("ros-melodic-geometry")
        .depend("ros-melodic-urdf")
        .depend("ros-melodic-control-msgs")
        .depend("ros-melodic-kdl-parser")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-ros-environment")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-ros-tutorials")
        .depend("ros-melodic-roscpp-tutorials")
        .depend("ros-melodic-rospy-tutorials")
        .depend("ros-melodic-turtlesim")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-roscpp-tutorials")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-roscpp-serialization")
        .depend("ros-melodic-message-runtime")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-roscpp-serialization")
        .make_depend("ros-melodic-rostime");
    raur.pkg("ros-melodic-roslang")
        .depend("ros-melodic-genmsg")
        .depend("ros-melodic-catkin")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-roslib")
        .depend("ros-melodic-ros-environment")
        .depend("ros-melodic-rospack")
        .depend("ros-melodic-catkin")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rospack")
        .make_depend("ros-melodic-catkin")
        .make_depend("boost");
    raur.pkg("ros-melodic-roslint")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rospack")
        .depend("ros-melodic-ros-environment")
        .depend("tinyxml2")
        .depend("python-rosdep")
        .depend("python-catkin_pkg")
        .depend("pkg-config")
        .depend("boost")
        .depend("python")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("tinyxml2")
        .make_depend("gtest")
        .make_depend("pkg-config")
        .make_depend("boost")
        .make_depend("python");
    raur.pkg("ros-melodic-rospy-tutorials")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rostest");
    raur.pkg("ros-melodic-rosunit")
        .depend("ros-melodic-roslib")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-common-plugins")
        .depend("ros-melodic-rqt-bag-plugins")
        .depend("ros-melodic-rqt-launch")
        .depend("ros-melodic-rqt-action")
        .depend("ros-melodic-rqt-msg")
        .depend("ros-melodic-rqt-logger-level")
        .depend("ros-melodic-rqt-top")
        .depend("ros-melodic-rqt-service-caller")
        .depend("ros-melodic-rqt-shell")
        .depend("ros-melodic-rqt-graph")
        .depend("ros-melodic-rqt-topic")
        .depend("ros-melodic-rqt-web")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-bag")
        .depend("ros-melodic-rqt-plot")
        .depend("ros-melodic-rqt-publisher")
        .depend("ros-melodic-rqt-console")
        .depend("ros-melodic-rqt-srv")
        .depend("ros-melodic-rqt-dep")
        .depend("ros-melodic-rqt-image-view")
        .depend("ros-melodic-rqt-py-console")
        .depend("ros-melodic-rqt-reconfigure")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-dep")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rqt-graph")
        .depend("ros-melodic-qt-gui-py-common")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-qt-dotgraph")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-image-view")
        .depend("ros-melodic-rqt-gui-cpp")
        .depend("ros-melodic-geometry-msgs")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rqt-gui-cpp")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-rqt-gui")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-image-transport")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-rqt-publisher")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-qt-gui-py-common")
        .depend("ros-melodic-rosmsg")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-roslib")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-py-console")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-qt-gui-py-common")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-reconfigure")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-rqt-console")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-srv")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rosmsg")
        .depend("ros-melodic-rqt-msg")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-self-test")
        .depend("ros-melodic-diagnostic-updater")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-diagnostic-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-diagnostic-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-diagnostic-updater");
    raur.pkg("ros-melodic-simulators")
        .depend("ros-melodic-rqt-common-plugins")
        .depend("ros-melodic-robot")
        .depend("ros-melodic-gazebo-ros-pkgs")
        .depend("ros-melodic-rqt-robot-plugins")
        .depend("ros-melodic-stage-ros")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-smach")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-smach-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-smach-ros")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-smach-msgs")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-actionlib")
        .depend("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-smach")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rostest");
    raur.pkg("ros-melodic-stage")
        .depend("ros-melodic-catkin")
        .depend("libjpeg-turbo")
        .depend("mesa")
        .depend("fltk")
        .depend("gtk2")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("fltk")
        .make_depend("libjpeg-turbo")
        .make_depend("gtk2")
        .make_depend("libtool")
        .make_depend("mesa")
        .make_depend("pkg-config");
    raur.pkg("ros-melodic-stage-ros")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-stage")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-geometry-msgs")
        .depend("ros-melodic-nav-msgs")
        .depend("ros-melodic-sensor-msgs")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-stage")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-std-srvs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-nav-msgs")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("boost");
    raur.pkg("ros-melodic-stereo-image-proc")
        .depend("ros-melodic-image-geometry")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-stereo-msgs")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-image-proc")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-image-transport")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-image-geometry")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-stereo-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-image-proc")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-image-transport");
    raur.pkg("ros-melodic-tf-conversions")
        .depend("ros-melodic-kdl-conversions")
        .depend("ros-melodic-python-orocos-kdl")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-geometry-msgs")
        .depend("eigen3")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-kdl-conversions")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("eigen3");
    raur.pkg("ros-melodic-tf2-eigen")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-geometry-msgs")
        .depend("eigen3")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf2")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("eigen3");
    raur.pkg("ros-melodic-tf2-geometry-msgs")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-python-orocos-kdl")
        .depend("ros-melodic-tf2-ros")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf2")
        .make_depend("ros-melodic-python-orocos-kdl")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf2-ros")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-theora-image-transport")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-image-transport")
        .depend("libtheora")
        .depend("libogg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-rosbag")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cv-bridge")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-image-transport")
        .make_depend("libtheora")
        .make_depend("libogg");
    raur.pkg("ros-melodic-transmission-interface")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-roscpp")
        .depend("tinyxml")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-hardware-interface")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("tinyxml");
    raur.pkg("ros-melodic-urdf-parser-plugin")
        .depend("urdfdom-headers")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("urdfdom-headers");
    raur.pkg("ros-melodic-urdf-sim-tutorial")
        .depend("ros-melodic-position-controllers")
        .depend("ros-melodic-controller-manager")
        .depend("ros-melodic-joint-state-controller")
        .depend("ros-melodic-diff-drive-controller")
        .depend("ros-melodic-urdf-tutorial")
        .depend("ros-melodic-gazebo-ros-control")
        .depend("ros-melodic-rqt-robot-steering")
        .depend("ros-melodic-gazebo-ros")
        .depend("ros-melodic-xacro")
        .depend("ros-melodic-rviz")
        .depend("ros-melodic-robot-state-publisher")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-urdfdom-py")
        .depend("python-yaml")
        .depend("python-lxml")
        .depend("python")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("python");
    raur.pkg("ros-melodic-vision-opencv")
        .depend("ros-melodic-cv-bridge")
        .depend("ros-melodic-image-geometry")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-viz")
        .depend("ros-melodic-rqt-robot-plugins")
        .depend("ros-melodic-ros-base")
        .depend("ros-melodic-rviz")
        .depend("ros-melodic-rqt-common-plugins")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("console-bridge")
        .depend("gcc-libs")
        .make_depend("cmake");
    raur.pkg("log4cxx")
        .provide("log4cxx")
        .depend("apr-util")
        .depend("libxml2")
        .make_depend("autoconf")
        .make_depend("automake")
        .make_depend("libtool")
        .make_depend("patch")
        .make_depend("zip")
        .make_depend("gzip")
        .make_depend("sed");
    raur.pkg("ogre-1.9")
        .provide("ogre=1.9")
        .provide("ogre-docs=1.9")
        .depend("freeimage")
        .depend("freetype2")
        .depend("libxaw")
        .depend("libxrandr")
        .depend("openexr")
        .depend("nvidia-cg-toolkit")
        .depend("zziplib")
        .depend("sdl2")
        .depend("glu")
        .depend("tinyxml")
        .make_depend("cmake")
        .make_depend("doxygen")
        .make_depend("graphviz")
        .make_depend("ttf-dejavu")
        .make_depend("mesa")
        .make_depend("python")
        .make_depend("swig")
        .make_depend("systemd");
    raur.pkg("ros-melodic-actionlib")
        .depend("ros-melodic-rostest")
        .depend("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .depend("wxpython")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-actionlib-msgs")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-rospy")
        .make_depend("boost");
    raur.pkg("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-message-generation")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-actionlib-tutorials")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-actionlib")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-actionlib")
        .make_depend("ros-melodic-actionlib-msgs")
        .make_depend("ros-melodic-roscpp");
    raur.pkg("ros-melodic-angles")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-bond")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-bond-core")
        .depend("ros-melodic-smclib")
        .depend("ros-melodic-bondpy")
        .depend("ros-melodic-bondcpp")
        .depend("ros-melodic-bond")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-bondcpp")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-smclib")
        .depend("ros-melodic-bond")
        .depend("boost")
        .depend("util-linux")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-smclib")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-bond")
        .make_depend("boost")
        .make_depend("util-linux");
    raur.pkg("ros-melodic-bondpy")
        .depend("ros-melodic-smclib")
        .depend("ros-melodic-rospy")
        .depend("util-linux")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-smclib")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-bond")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-class-loader")
        .depend("boost")
        .depend("console-bridge")
        .depend("poco")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("boost")
        .make_depend("console-bridge")
        .make_depend("poco");
    raur.pkg("ros-melodic-common-msgs")
        .depend("ros-melodic-diagnostic-msgs")
        .depend("ros-melodic-trajectory-msgs")
        .depend("ros-melodic-stereo-msgs")
        .depend("ros-melodic-nav-msgs")
        .depend("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-shape-msgs")
        .depend("ros-melodic-visualization-msgs")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-common-tutorials")
        .depend("ros-melodic-pluginlib-tutorials")
        .depend("ros-melodic-turtle-actionlib")
        .depend("ros-melodic-nodelet-tutorial-math")
        .depend("ros-melodic-actionlib-tutorials")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-cpp-common")
        .depend("boost")
        .depend("console-bridge")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("boost")
        .make_depend("console-bridge");
    raur.pkg("ros-melodic-diagnostic-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-rosservice")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-roscpp-serialization")
        .make_depend("boost");
    raur.pkg("ros-melodic-gencpp")
        .depend("ros-melodic-genmsg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-genmsg")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-geneus")
        .depend("ros-melodic-genmsg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-genmsg")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-genlisp")
        .depend("ros-melodic-genmsg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-genmsg")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-genmsg")
        .depend("ros-melodic-catkin")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-gennodejs")
        .depend("ros-melodic-genmsg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-genmsg")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-genpy")
        .depend("ros-melodic-genmsg")
        .depend("python-yaml")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-genmsg")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-geometry-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-geometry-tutorials")
        .depend("ros-melodic-turtle-tf")
        .depend("ros-melodic-turtle-tf2")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-gl-dependency")
        .depend("python-pyqt5")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-image-transport")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-roslib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-roslib");
    raur.pkg("ros-melodic-interactive-marker-tutorials")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-visualization-msgs")
        .depend("ros-melodic-interactive-markers")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-interactive-markers")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-visualization-msgs");
    raur.pkg("ros-melodic-interactive-markers")
        .depend("ros-melodic-rostest")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-visualization-msgs")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-visualization-msgs")
        .make_depend("ros-melodic-rospy");
    raur.pkg("ros-melodic-joint-state-publisher")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-kdl-parser")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-urdf")
        .depend("ros-melodic-rosconsole")
        .depend("tinyxml")
        .depend("tinyxml2")
        .depend("urdfdom-headers")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-urdf")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("tinyxml")
        .make_depend("tinyxml2")
        .make_depend("urdfdom-headers");
    raur.pkg("ros-melodic-laser-geometry")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-angles")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-roscpp")
        .depend("boost")
        .depend("eigen3")
        .depend("python-numpy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-angles")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("boost")
        .make_depend("eigen3");
    raur.pkg("ros-melodic-librviz-tutorial")
        .depend("ros-melodic-rviz")
        .depend("ros-melodic-roscpp")
        .depend("qt5-base")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rviz")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-map-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-nav-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-nav-msgs")
        .make_depend("ros-melodic-sensor-msgs");
    raur.pkg("ros-melodic-media-export")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-message-filters")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-rosunit")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("boost");
    raur.pkg("ros-melodic-message-generation")
        .depend("ros-melodic-geneus")
        .depend("ros-melodic-gencpp")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-gennodejs")
        .depend("ros-melodic-genlisp")
        .depend("ros-melodic-genmsg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-message-runtime")
        .depend("ros-melodic-roscpp-traits")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-cpp-common")
        .depend("ros-melodic-roscpp-serialization")
        .depend("ros-melodic-rostime")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-mk")
        .depend("ros-melodic-rosbuild")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-nav-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-actionlib-msgs")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-nodelet")
        .depend("ros-melodic-bondcpp")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-pluginlib")
        .depend("boost")
        .depend("util-linux")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-bondcpp")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("boost")
        .make_depend("util-linux");
    raur.pkg("ros-melodic-nodelet-core")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-nodelet-topic-tools")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-nodelet-topic-tools")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-pluginlib")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-dynamic-reconfigure")
        .make_depend("ros-melodic-catkin")
        .make_depend("boost");
    raur.pkg("ros-melodic-nodelet-tutorial-math")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-orocos-kdl")
        .depend("ros-melodic-catkin")
        .depend("eigen3")
        .depend("pkg-config")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("eigen3");
    raur.pkg("ros-melodic-pluginlib")
        .depend("ros-melodic-class-loader")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roslib")
        .depend("boost")
        .depend("tinyxml2")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-class-loader")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roslib")
        .make_depend("boost")
        .make_depend("tinyxml2");
    raur.pkg("ros-melodic-pluginlib-tutorials")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-python-qt-binding")
        .depend("python-pyqt5")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rosbuild")
        .make_depend("ros-melodic-catkin")
        .make_depend("python-pyqt5")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-qt-dotgraph")
        .depend("ros-melodic-python-qt-binding")
        .depend("python-pydot")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-qt-gui")
        .depend("ros-melodic-python-qt-binding")
        .depend("python-rospkg")
        .depend("tango-icon-theme")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("python-pyqt5")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-qt-gui-cpp")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-pluginlib")
        .depend("tinyxml")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-python-qt-binding")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("tinyxml")
        .make_depend("qt5-base")
        .make_depend("pkg-config");
    raur.pkg("ros-melodic-qt-gui-py-common")
        .depend("ros-melodic-python-qt-binding")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-qwt-dependency")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-resource-retriever")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roslib")
        .depend("boost")
        .depend("python-rospkg")
        .depend("curl")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-roslib")
        .make_depend("boost")
        .make_depend("curl");
    raur.pkg("ros-melodic-robot-state-publisher")
        .depend("ros-melodic-tf2-kdl")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-catkin")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-tf2-ros")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-kdl-parser")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-sensor-msgs")
        .depend("eigen3")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf2-kdl")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-tf2-ros")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-kdl-parser")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("eigen3")
        .make_depend("urdfdom-headers");
    raur.pkg("ros-melodic-ros")
        .depend("ros-melodic-rosmake")
        .depend("ros-melodic-rosboost-cfg")
        .depend("ros-melodic-rosbuild")
        .depend("ros-melodic-rosclean")
        .depend("ros-melodic-rosbash")
        .depend("ros-melodic-catkin")
        .depend("ros-melodic-rosunit")
        .depend("ros-melodic-mk")
        .depend("ros-melodic-roscreate")
        .depend("ros-melodic-roslang")
        .depend("ros-melodic-roslib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-ros-base")
        .depend("ros-melodic-bond-core")
        .depend("ros-melodic-nodelet-core")
        .depend("ros-melodic-dynamic-reconfigure")
        .depend("ros-melodic-ros-core")
        .depend("ros-melodic-actionlib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-ros-comm")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-rosmsg")
        .depend("ros-melodic-rosout")
        .depend("ros-melodic-rosparam")
        .depend("ros-melodic-topic-tools")
        .depend("ros-melodic-rosgraph")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-ros")
        .depend("ros-melodic-roslisp")
        .depend("ros-melodic-roswtf")
        .depend("ros-melodic-rosmaster")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-xmlrpcpp")
        .depend("ros-melodic-rostest")
        .depend("ros-melodic-rosnode")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-roslaunch")
        .depend("ros-melodic-rosservice")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-ros-core")
        .depend("ros-melodic-geneus")
        .depend("ros-melodic-roscpp-core")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-gencpp")
        .depend("ros-melodic-rosconsole-bridge")
        .depend("ros-melodic-genmsg")
        .depend("ros-melodic-rosbag-migration-rule")
        .depend("ros-melodic-genlisp")
        .depend("ros-melodic-message-generation")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-ros")
        .depend("ros-melodic-roslisp")
        .depend("ros-melodic-class-loader")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-cmake-modules")
        .depend("ros-melodic-rospack")
        .depend("ros-melodic-catkin")
        .depend("ros-melodic-common-msgs")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-gennodejs")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-ros-comm")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-pluginlib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosbag")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-xmlrpcpp")
        .depend("ros-melodic-rosbag-storage")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-topic-tools")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-genmsg")
        .depend("ros-melodic-roslib")
        .depend("python-rospkg")
        .depend("boost")
        .depend("python-gnupg")
        .depend("python-crypto")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-std-srvs")
        .make_depend("ros-melodic-xmlrpcpp")
        .make_depend("ros-melodic-rosbag-storage")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-roscpp-serialization")
        .make_depend("ros-melodic-topic-tools")
        .make_depend("python-pillow")
        .make_depend("boost");
    raur.pkg("ros-melodic-rosbag-migration-rule")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosbag-storage")
        .depend("ros-melodic-roscpp-traits")
        .depend("ros-melodic-roslz4")
        .depend("ros-melodic-cpp-common")
        .depend("ros-melodic-roscpp-serialization")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-pluginlib")
        .depend("console-bridge")
        .depend("gpgme")
        .depend("openssl")
        .depend("boost")
        .depend("bzip2")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-roscpp-traits")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-roslz4")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-roscpp-serialization")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("console-bridge")
        .make_depend("gpgme")
        .make_depend("openssl")
        .make_depend("boost")
        .make_depend("bzip2");
    raur.pkg("ros-melodic-rosbash")
        .depend("ros-melodic-catkin")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosboost-cfg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosbuild")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-message-generation")
        .depend("ros-melodic-catkin")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("pkg-config");
    raur.pkg("ros-melodic-rosclean")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosconsole")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-cpp-common")
        .depend("ros-melodic-rosbuild")
        .depend("log4cxx")
        .depend("apr")
        .depend("apr-util")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-rosunit")
        .make_depend("ros-melodic-catkin")
        .make_depend("log4cxx")
        .make_depend("apr")
        .make_depend("boost")
        .make_depend("apr-util");
    raur.pkg("ros-melodic-rosconsole-bridge")
        .depend("ros-melodic-rosconsole")
        .depend("console-bridge")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-catkin")
        .make_depend("console-bridge");
    raur.pkg("ros-melodic-roscpp")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-xmlrpcpp")
        .depend("ros-melodic-roscpp-traits")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-cpp-common")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp-serialization")
        .depend("ros-melodic-message-runtime")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-xmlrpcpp")
        .make_depend("ros-melodic-roscpp-traits")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rosgraph-msgs")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp-serialization")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-roslang")
        .make_depend("pkg-config");
    raur.pkg("ros-melodic-roscpp-core")
        .depend("ros-melodic-roscpp-traits")
        .depend("ros-melodic-cpp-common")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-roscpp-serialization")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-roscpp-serialization")
        .depend("ros-melodic-roscpp-traits")
        .depend("ros-melodic-cpp-common")
        .depend("ros-melodic-rostime")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-roscpp-traits")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-roscpp-traits")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-cpp-common")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-roscreate")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosgraph")
        .depend("python-yaml")
        .depend("python-netifaces")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-roslaunch")
        .depend("ros-melodic-rosout")
        .depend("ros-melodic-rosparam")
        .depend("ros-melodic-rosclean")
        .depend("ros-melodic-rosunit")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-rosmaster")
        .depend("ros-melodic-roslib")
        .depend("python-yaml")
        .depend("python-paramiko")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-roslisp")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-roslang")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-rospack")
        .depend("sbcl")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-roslz4")
        .depend("lz4")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("lz4");
    raur.pkg("ros-melodic-rosmake")
        .depend("ros-melodic-catkin")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosmaster")
        .depend("ros-melodic-rosgraph")
        .depend("python-defusedxml")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosmsg")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-catkin")
        .depend("ros-melodic-genmsg")
        .depend("ros-melodic-roslib")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosnode")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-rosgraph")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rostest");
    raur.pkg("ros-melodic-rosout")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rosgraph-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosparam")
        .depend("ros-melodic-rosgraph")
        .depend("python-yaml")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rospy")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-rosgraph")
        .depend("ros-melodic-roslib")
        .depend("python-yaml")
        .depend("python-rospkg")
        .depend("python-numpy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rosservice")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-rosmsg")
        .depend("ros-melodic-rosgraph")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rostest")
        .depend("ros-melodic-rosunit")
        .depend("ros-melodic-rosgraph")
        .depend("ros-melodic-rosmaster")
        .depend("ros-melodic-roslaunch")
        .depend("ros-melodic-rospy")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rosunit")
        .make_depend("ros-melodic-catkin")
        .make_depend("boost");
    raur.pkg("ros-melodic-rostime")
        .depend("ros-melodic-cpp-common")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-catkin")
        .make_depend("boost");
    raur.pkg("ros-melodic-rostopic")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-rosbag")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rostest");
    raur.pkg("ros-melodic-roswtf")
        .depend("ros-melodic-rosbuild")
        .depend("ros-melodic-rosnode")
        .depend("ros-melodic-rosgraph")
        .depend("ros-melodic-roslaunch")
        .depend("ros-melodic-rosservice")
        .depend("ros-melodic-roslib")
        .depend("python-paramiko")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rostest");
    raur.pkg("ros-melodic-rqt-action")
        .depend("ros-melodic-rqt-msg")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-bag")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-rosnode")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-bag-plugins")
        .depend("ros-melodic-rqt-plot")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-geometry-msgs")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-rqt-bag")
        .depend("ros-melodic-roslib")
        .depend("python-pillow")
        .depend("python-cairo")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-console")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rqt-logger-level")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-graph")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rosservice")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rosnode")
        .depend("ros-melodic-rosgraph-msgs")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-rosgraph")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-qt-dotgraph")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-gui")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-catkin")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-qt-gui")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-gui-cpp")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-nodelet")
        .depend("ros-melodic-qt-gui-cpp")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-nodelet")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-qt-gui")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-qt-gui-cpp")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-qt-gui")
        .make_depend("ros-melodic-rqt-gui")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-launch")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rqt-console")
        .depend("ros-melodic-roslaunch")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rqt-py-common")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-logger-level")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rosservice")
        .depend("ros-melodic-rosnode")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-moveit")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rqt-topic")
        .depend("ros-melodic-rosnode")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-msg")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rosmsg")
        .depend("ros-melodic-rqt-console")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-nav-view")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-nav-msgs")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-plot")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-qwt-dependency")
        .depend("ros-melodic-qt-gui-py-common")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-rosgraph")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("python-rospkg")
        .depend("python-matplotlib")
        .depend("python-numpy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-pose-view")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-gl-dependency")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-geometry-msgs")
        .depend("python-opengl")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-py-common")
        .depend("ros-melodic-genpy")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-actionlib")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-genmsg")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-robot-dashboard")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-diagnostic-msgs")
        .depend("ros-melodic-rqt-nav-view")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-rqt-console")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-rqt-robot-monitor")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-robot-monitor")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-diagnostic-msgs")
        .depend("ros-melodic-qt-gui-py-common")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-rqt-bag")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rospy")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-robot-plugins")
        .depend("ros-melodic-rqt-tf-tree")
        .depend("ros-melodic-rqt-runtime-monitor")
        .depend("ros-melodic-rqt-nav-view")
        .depend("ros-melodic-rqt-pose-view")
        .depend("ros-melodic-rqt-robot-steering")
        .depend("ros-melodic-rqt-moveit")
        .depend("ros-melodic-rqt-robot-dashboard")
        .depend("ros-melodic-rqt-rviz")
        .depend("ros-melodic-rqt-robot-monitor")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-robot-steering")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-geometry-msgs")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-runtime-monitor")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-diagnostic-msgs")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-rviz")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rqt-gui-cpp")
        .depend("ros-melodic-rviz")
        .depend("ros-melodic-pluginlib")
        .depend("boost")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rviz")
        .make_depend("ros-melodic-rqt-gui-cpp")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rqt-gui")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("boost")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-rqt-service-caller")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rqt-py-common")
        .depend("ros-melodic-rosservice")
        .depend("ros-melodic-rqt-gui-py")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-shell")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-qt-gui-py-common")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-tf-tree")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-rqt-graph")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-tf2-ros")
        .depend("ros-melodic-tf2-msgs")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-qt-dotgraph")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-geometry-msgs")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-top")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-rqt-gui-py")
        .depend("python-psutil")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-topic")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rostopic")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rqt-web")
        .depend("ros-melodic-rqt-gui-py")
        .depend("ros-melodic-qt-gui")
        .depend("ros-melodic-webkit-dependency")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-rqt-gui")
        .depend("ros-melodic-rospy")
        .depend("python-rospkg")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-rviz")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-rosbag")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-nav-msgs")
        .depend("ros-melodic-urdf")
        .depend("ros-melodic-python-qt-binding")
        .depend("ros-melodic-resource-retriever")
        .depend("ros-melodic-laser-geometry")
        .depend("ros-melodic-media-export")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-interactive-markers")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-roslib")
        .depend("ros-melodic-image-transport")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-visualization-msgs")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-map-msgs")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-geometry-msgs")
        .depend("yaml-cpp")
        .depend("eigen3")
        .depend("ogre-1.9")
        .depend("assimp")
        .depend("mesa")
        .depend("tinyxml2")
        .depend("urdfdom-headers")
        .depend("qt5-base")
        .depend("sip")
        .depend("python-sip")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-std-srvs")
        .make_depend("ros-melodic-rosbag")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-nav-msgs")
        .make_depend("ros-melodic-urdf")
        .make_depend("ros-melodic-python-qt-binding")
        .make_depend("ros-melodic-resource-retriever")
        .make_depend("ros-melodic-laser-geometry")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-interactive-markers")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-roslib")
        .make_depend("ros-melodic-image-transport")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-visualization-msgs")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-map-msgs")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("yaml-cpp")
        .make_depend("eigen3")
        .make_depend("ogre-1.9")
        .make_depend("assimp")
        .make_depend("mesa")
        .make_depend("tinyxml2")
        .make_depend("urdfdom-headers")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-rviz-plugin-tutorials")
        .depend("ros-melodic-rviz")
        .depend("qt5-base")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rviz")
        .make_depend("ros-melodic-catkin")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-rviz-python-tutorial")
        .depend("ros-melodic-rviz")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rviz")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-sensor-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-shape-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-smclib")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-std-msgs")
        .depend("ros-melodic-message-runtime")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-std-srvs")
        .depend("ros-melodic-message-runtime")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-stereo-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-std-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-std-msgs");
    raur.pkg("ros-melodic-tf")
        .depend("ros-melodic-roswtf")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-tf2-ros")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-sensor-msgs")
        .depend("ros-melodic-geometry-msgs")
        .depend("graphviz")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-angles")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-tf2-ros")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-sensor-msgs")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-tf2")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-tf2-msgs")
        .depend("ros-melodic-geometry-msgs")
        .depend("console-bridge")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-tf2-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("console-bridge");
    raur.pkg("ros-melodic-tf2-kdl")
        .depend("ros-melodic-orocos-kdl")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-tf2-ros")
        .depend("eigen3")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf2")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf2-ros")
        .make_depend("ros-melodic-orocos-kdl")
        .make_depend("eigen3");
    raur.pkg("ros-melodic-tf2-msgs")
        .depend("ros-melodic-message-generation")
        .depend("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-actionlib-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-tf2-py")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-rospy")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf2")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-tf2-ros")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-xmlrpcpp")
        .depend("ros-melodic-tf2-py")
        .depend("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-actionlib")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-rosgraph")
        .depend("ros-melodic-message-filters")
        .depend("ros-melodic-tf2-msgs")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-tf2")
        .make_depend("ros-melodic-xmlrpcpp")
        .make_depend("ros-melodic-tf2-py")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-actionlib-msgs")
        .make_depend("ros-melodic-actionlib")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-rosgraph")
        .make_depend("ros-melodic-message-filters")
        .make_depend("ros-melodic-tf2-msgs")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-topic-tools")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-xmlrpcpp")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-message-runtime")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-xmlrpcpp")
        .make_depend("ros-melodic-rostest")
        .make_depend("ros-melodic-rosunit")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-rostime");
    raur.pkg("ros-melodic-trajectory-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rosbag-migration-rule")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-turtle-actionlib")
        .depend("ros-melodic-angles")
        .depend("ros-melodic-turtlesim")
        .depend("ros-melodic-actionlib-msgs")
        .depend("ros-melodic-actionlib")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-angles")
        .make_depend("ros-melodic-turtlesim")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-actionlib-msgs")
        .make_depend("ros-melodic-actionlib")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-turtle-tf")
        .depend("ros-melodic-turtlesim")
        .depend("ros-melodic-tf")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-turtlesim")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-tf")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-turtle-tf2")
        .depend("ros-melodic-turtlesim")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-tf2-ros")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-tf2")
        .depend("ros-melodic-rospy")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-turtlesim")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-tf2-ros")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-tf2")
        .make_depend("ros-melodic-rospy")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-turtlesim")
        .depend("ros-melodic-std-srvs")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-roslib")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-rosconsole")
        .depend("ros-melodic-roscpp")
        .depend("ros-melodic-roscpp-serialization")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-geometry-msgs")
        .depend("qt5-base")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-std-srvs")
        .make_depend("ros-melodic-roslib")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-rosconsole")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-roscpp-serialization")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-geometry-msgs")
        .make_depend("qt5-base");
    raur.pkg("ros-melodic-urdf")
        .depend("ros-melodic-rosconsole-bridge")
        .depend("ros-melodic-pluginlib")
        .depend("ros-melodic-roscpp")
        .depend("tinyxml")
        .depend("tinyxml2")
        .depend("urdfdom")
        .depend("urdfdom-headers")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-urdf-parser-plugin")
        .make_depend("ros-melodic-cmake-modules")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-rosconsole-bridge")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-pluginlib")
        .make_depend("tinyxml")
        .make_depend("tinyxml2")
        .make_depend("urdfdom")
        .make_depend("urdfdom-headers");
    raur.pkg("ros-melodic-urdf-tutorial")
        .depend("ros-melodic-xacro")
        .depend("ros-melodic-joint-state-publisher")
        .depend("ros-melodic-rviz")
        .depend("ros-melodic-robot-state-publisher")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-roslaunch");
    raur.pkg("ros-melodic-visualization-marker-tutorials")
        .depend("ros-melodic-visualization-msgs")
        .depend("ros-melodic-roscpp")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-visualization-msgs")
        .make_depend("ros-melodic-roscpp")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-visualization-msgs")
        .depend("ros-melodic-message-runtime")
        .depend("ros-melodic-std-msgs")
        .depend("ros-melodic-geometry-msgs")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-message-generation")
        .make_depend("ros-melodic-std-msgs")
        .make_depend("ros-melodic-catkin")
        .make_depend("ros-melodic-geometry-msgs");
    raur.pkg("ros-melodic-visualization-tutorials")
        .depend("ros-melodic-visualization-marker-tutorials")
        .depend("ros-melodic-librviz-tutorial")
        .depend("ros-melodic-rviz-plugin-tutorials")
        .depend("ros-melodic-interactive-marker-tutorials")
        .depend("ros-melodic-rviz-python-tutorial")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-webkit-dependency")
        .depend("python-pyqt5")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-xacro")
        .depend("ros-melodic-roslaunch")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-roslint")
        .make_depend("ros-melodic-catkin");
    raur.pkg("ros-melodic-xmlrpcpp")
        .depend("ros-melodic-rostime")
        .depend("ros-melodic-cpp-common")
        .make_depend("cmake")
        .make_depend("ros-build-tools")
        .make_depend("ros-melodic-rostime")
        .make_depend("ros-melodic-cpp-common")
        .make_depend("ros-melodic-catkin");
    raur.pkg("tango-icon-theme")
        .make_depend("imagemagick")
        .make_depend("icon-naming-utils")
        .make_depend("intltool")
        .make_depend("librsvg");
    raur.pkg("urdfdom")
        .depend("tinyxml")
        .depend("console-bridge")
        .depend("urdfdom-headers")
        .make_depend("cmake");
    raur.pkg("urdfdom-headers").make_depend("cmake");

    raur.pkg("yay")
        .depend("pacman>=5.1")
        .depend("sudo")
        .depend("git")
        .make_depend("go");
    raur.pkg("yay-bin")
        .depend("pacman>=5.1")
        .depend("sudo")
        .depend("git")
        .make_depend("go")
        .provide("yay")
        .conflict("yay");
    raur.pkg("yay-git")
        .depend("pacman>=5.1")
        .depend("sudo")
        .depend("git")
        .make_depend("go")
        .provide("yay")
        .conflict("yay");

    raur.pkg("auracle-git")
        .provide("auracle")
        .depend("pacman")
        .depend("libarchive.so")
        .depend("libcurl.so")
        .depend("libsystemd.so")
        .make_depend("meson")
        .make_depend("git")
        .make_depend("perl");
    raur.pkg("pacaur")
        .depend("auracle-git")
        .depend("expac")
        .depend("sudo")
        .depend("git")
        .depend("jq")
        .make_depend("perl")
        .make_depend("git");

    raur.pkg("spotify")
        .depend("alsa-lib>=1.0.14")
        .depend("gconf")
        .depend("gtk2")
        .depend("glib2")
        .depend("nss")
        .depend("libsystemd")
        .depend("libxtst")
        .depend("libx11")
        .depend("libxss")
        .depend("desktop-file-utils")
        .depend("rtmpdump")
        .depend("openssl-1.0")
        .depend("libcurl-gnutls");

    raur.pkg("discord-canary")
        .version("0.0.96-1")
        .provide("discord")
        .make_depend("libc++")
        .depend("gtk3")
        .depend("libnotify")
        .depend("libxss")
        .depend("glibc")
        .depend("alsa-lib")
        .depend("nspr")
        .depend("nss")
        .depend("xdg-utils")
        .depend("libcups");
    raur.pkg("libc++")
        .version("8.0.1-1")
        .depend("libc++abi=8.0.1-1")
        .make_depend("clang")
        .make_depend("cmake")
        .make_depend("ninja")
        .make_depend("python")
        .make_depend("libunwind");
    raur.pkg("libc++abi")
        .version("8.0.1-1")
        .depend("gcc-libs")
        .make_depend("clang")
        .make_depend("cmake")
        .make_depend("ninja")
        .make_depend("python")
        .make_depend("libunwind");

    raur.pkg("pacman-git")
        .version("5.1.1.r160.gd37e6d40-2")
        .provide("pacman=5.1.1")
        .depend("archlinux-keyring")
        .depend("bash")
        .depend("curl")
        .depend("gpgme")
        .depend("libarchive")
        .depend("pacman-mirrorlist")
        .make_depend("git")
        .make_depend("asciidoc")
        .make_depend("meson")
        .conflict("pacman");

    raur.pkg("assume-test")
        .depend("assume-dep1")
        .depend("assume-dep2")
        .depend("i3-wm")
        .depend("libev");
    raur.pkg("assume-dep1");
    raur.pkg("assume-dep2");

    raur.pkg("repo_version_test").depend("pacman-contrib>100");
    raur.pkg("satisfied_versioned_repo_dep")
        .depend("pacman>100");

    raur.pkg("version_equal").version("1-1");
    raur.pkg("version_newer").version("100-1");
    raur.pkg("version_older").version("0-1");

    raur.pkg("cyclic-dep").depend("cyclic-dep");
    raur.pkg("cyclic").depend("cyclic-dep");

    raur.pkg("xterm");

    raur.pkg("systemd-git").depend("systemd").provide("systemd");

    raur
}
