pub use generators::*;
pub use util::rms_level;

pub mod charts;
mod generators;
mod util;

#[macro_export]
macro_rules! assert_f_eq {
    ($left:expr, $right:expr) => {{
        match (&$left, &$right) {
            (left_val, right_val) => {
                assert!(
                    ((left_val - right_val).abs() as f32) < f32::EPSILON,
                    "left: {:?} not equal right: {:?}",
                    left_val,
                    right_val
                );
            }
        }
    }};
}

#[macro_export]
macro_rules! relative_path {
    ($path: expr) => {
        format!("{}/./{}", env!("CARGO_MANIFEST_DIR"), $path)
    };
}

/// Test two buffers have equivalent RMS levels
pub fn test_level_equivalence(
    input_buffer: &[f32],
    output_buffer: &[f32],
    input_window_size: usize,
    output_window_size: usize,
    threshold: f32,
) {
    let input_chunks = input_buffer.chunks(input_window_size);
    let output_chunks = output_buffer.chunks(output_window_size);
    assert!(!input_buffer.is_empty());
    assert!(!output_buffer.is_empty());
    // assert!((input_chunks.len() as i32 - output_chunks.len() as i32).abs() < 2);
    for (input_chunk, output_chunk) in input_chunks.zip(output_chunks) {
        let input_level = util::rms_level(input_chunk);
        let output_level = util::rms_level(output_chunk);
        let diff = (input_level - output_level).abs();

        assert!(diff < threshold);
    }
}

#[cfg(test)]
mod test {
    #[test]
    fn test_relative_path() {
        assert!(std::fs::read_to_string(relative_path!("src/lib.rs"))
            .unwrap()
            .contains("fn test_relative_path()"));
    }
}
