//! Content-type / Accept header negotiation, MIME types

use actix_web::{http::HeaderMap};

#[derive(PartialEq)]
pub enum ContentType {
    /// Plain JSON, using shortnames as keys instead of URLs
    /// https://docs.atomicdata.dev/interoperability/json.html#atomic-data-as-plain-json
    JSON,
    /// JSON-AD, default Atomic Data serialization
    /// https://docs.atomicdata.dev/core/json-ad.html
    JSONAD,
    /// JSON-LD, RDF compatible JSON with @context mapping
    /// https://docs.atomicdata.dev/interoperability/json.html#from-json-to-json-ad
    JSONLD,
    HTML,
    /// RDF Turtle format
    /// https://www.w3.org/TR/turtle/
    TURTLE,
    /// RDF N-Triples format
    /// https://www.w3.org/TR/n-triples/
    NT,
}

const MIME_HTML: &str = "text/html";
const MIME_XML: &str = "application/xml";
const MIME_JSON: &str = "application/json";
const MIME_JSONLD: &str = "application/ld+json";
const MIME_JSONAD: &str = "application/ad+json";
const MIME_TURTLE: &str = "text/turtle";
const MIME_NT: &str = "application/n-triples";

impl ContentType {
    pub fn to_mime(&self) -> &str {
        match self {
            ContentType::JSON => MIME_JSON,
            ContentType::JSONAD => MIME_JSONAD,
            ContentType::JSONLD => MIME_JSONLD,
            ContentType::HTML => MIME_HTML,
            ContentType::TURTLE => MIME_TURTLE,
            ContentType::NT => MIME_NT
        }
    }
}

/// Returns the preffered content type.
/// Defaults to HTML if none is found.
pub fn get_accept(map: &HeaderMap) -> ContentType {
    let accept_header = match map.get("Accept") {
        Some(header) => {
            header.to_str().unwrap_or("")
        }
        None => {
            return ContentType::HTML
        }
    };
    parse_accept_header(accept_header)
}

/// Parses an HTTP Accept header
/// Does not fully adhere to the RFC spec: https://tools.ietf.org/html/rfc7231
/// Does not take into consideration the q value, simply reads the first thing before the comma
/// Defaults to HTML
pub fn parse_accept_header(header: &str) -> ContentType {
    for mimepart in header.split(',') {
        if mimepart.contains(MIME_JSONAD) {
            return ContentType::JSONAD
        }
        if mimepart.contains(MIME_HTML) {
            return ContentType::HTML
        }
        if mimepart.contains(MIME_XML) {
            return ContentType::HTML
        }
        if mimepart.contains(MIME_JSON) {
            return ContentType::JSON
        }
        if mimepart.contains(MIME_JSONLD) {
            return ContentType::JSONLD
        }
        if mimepart.contains(MIME_TURTLE) {
            return ContentType::TURTLE
        }
        if mimepart.contains(MIME_NT) {
            return ContentType::NT
        }
    }
    log::info!("Unknown Accept header, defaut to HTML: {}", header);
    ContentType::HTML
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn parse_types() {
        assert!(parse_accept_header("text/html,application/xml") == ContentType::HTML);
        assert!(parse_accept_header("application/ad+json") == ContentType::JSONAD);
        assert!(parse_accept_header("application/ld+json") == ContentType::JSONLD);
    }

    #[test]
    fn parse_types_with_blank_chars() {
        assert!(parse_accept_header("application/ad+json ; ") == ContentType::JSONAD);
        assert!(parse_accept_header(" application/ad+json ; ") == ContentType::JSONAD);
    }
}
