use std::collections::HashMap;
use glob::glob;
use std::fs;
use serde_json::Value;
use termion::{color, style};

pub struct EntityChecker {
    all_qualified_names: HashMap<String, bool>,
    referenced_qualified_names: HashMap<String, bool>,
}

impl EntityChecker {
    pub fn new(file_prefix: &str) -> EntityChecker {
        let mut ec = EntityChecker {
            all_qualified_names: HashMap::new(),
            referenced_qualified_names: HashMap::new(),
        };

        for path in glob(file_prefix).expect("Failed to read file_prefix") {
            match path {
                Ok(path) => {
                    println!("- Loading File: {:?}", path.to_str().unwrap());
                    let data = fs::read_to_string(path).unwrap();
                    let mut lines: Vec<&str> = data.split('\n').collect();
                    lines.remove(lines.len() - 1);

                    for line in lines {
                        let entity: Value = serde_json::from_str(line).unwrap();
                        ec.all_qualified_names.insert(entity["attributes"]["qualifiedName"].to_string(), true);

                        for (k, _) in entity["attributes"].as_object().unwrap().iter() {
                            if entity["attributes"][k].is_object() {
                                let qname = entity["attributes"][k]["uniqueAttributes"]["qualifiedName"].to_string();
                                ec.referenced_qualified_names.insert(qname, true)
                            } else {
                                continue;
                            };
                        }
                    }
                }
                Err(e) => eprintln!("{:?}", e)
            }
        }

        println!("\n{}{}Total Qualified Names: {:?}", style::Bold, color::Fg(color::Yellow), ec.all_qualified_names.len());
        println!("Referenced Qualified Names: {:?}{}\n", ec.referenced_qualified_names.len(), style::Reset);

        return ec;
    }

    pub fn run_checks(&self) {
        println!("{}{}{}Checking Referenced Qualified Names{}\n", style::Bold, color::Bg(color::Blue), color::Fg(color::White), style::Reset);
        let mut error_count = 0;
        let mut correct_count = 0;
        for (k, _) in &self.referenced_qualified_names {
            if &self.all_qualified_names.contains_key(k) == &false {
                error_count += 1;
                println!("{}No reference to qualifiedName {:?}", color::Fg(color::Red), k);
            } else {
                correct_count += 1;
            };
        }
        println!("\n{}{}Reference Check Completed{}", style::Bold, color::Fg(color::Yellow), style::Reset);
        println!("❌ Invalid References: {:?}", error_count);
        println!("✅ Valid References: {:?}", correct_count);

    }
}