use crc16::MODBUS;

use self::ac_control_message::ACControlMessage;
use self::ac_status_message::ACStatus;
use self::group_control_message::GroupControlMessage;
use self::group_status_message::GroupStatus;

pub mod ac_control_message;
pub mod ac_status_message;
pub mod group_control_message;
pub mod group_status_message;

pub struct MessageType;

impl MessageType {
    pub const GROUP_CONTROL_MESSAGE: u8 = 0x2A;
    pub const GROUP_STATUS_MESSAGE: u8 = 0x2B;
    pub const AC_CONTROL_MESSAGE: u8 = 0x2C;
    pub const AC_STATUS_MESSAGE: u8 = 0x2D;
}

#[derive(Debug)]
pub enum Message {
    /// Group control messages are to control all groups.
    /// Each message to AirTouch is to control one specific group.
    GroupControl(GroupControlMessage),
    /// The group status for all groups.
    GroupStatuses(Vec<GroupStatus>),
    /// Control a specific AC unit.
    /// Each message to AirTouch is to control one specific AC unit.
    ACControl(ACControlMessage),
    /// The AC status for all AC units.
    ACStatuses(Vec<ACStatus>),
}

/// Parses a byte array into the relevant message.
pub fn parse_message(bytes: &[u8]) -> Result<Message, &'static str> {
    if bytes.len() < 10 {
        return Err("bytes length is too short to be a valid AT4 message.");
    }

    // Check header bytes
    if bytes[0] != 0x55 || bytes[1] != 0x55 {
        return Err("invalid header.");
    }

    // Check address byte
    if bytes[3] != 0x80 && bytes[3] != 0x90 {
        return Err("invalid address byte.");
    }

    // bytes 7, 8
    let message_length: usize = (((bytes[6] as u16) << 2) | (bytes[7] as u16)) as usize;
    if bytes.len() < (message_length + 10) {
        return Err("bytes length is not long enough for parsed length.");
    }

    // data to be checked by the crc algorithm
    let data = bytes.get(2..(8 + message_length)).unwrap();

    // crc values to check against
    let crc = bytes
        .get((8 + message_length)..(8 + message_length + 2))
        .unwrap();

    let crc: u16 = (crc[0] as u16) << 2 | (crc[1] as u16);

    // validate crc
    if crc16::State::<MODBUS>::calculate(data) != crc {
        return Err("crc check failed.");
    }

    // TODO: Add the message ID to the return messages

    // Check according to message type
    match bytes[5] {
        // Group status
        MessageType::GROUP_STATUS_MESSAGE => todo!(),
        // AC status
        MessageType::AC_STATUS_MESSAGE => todo!(),
        _ => return Err("invalid message type."),
    }
}

#[cfg(test)]
mod tests;
