use crc16::MODBUS;

use crate::states;

struct MessageType;

impl MessageType {
    pub const GROUP_CONTROL_MESSAGE: u8 = 0x2A;
    pub const GROUP_STATUS_MESSAGE: u8 = 0x2B;
    pub const AC_CONTROL_MESSAGE: u8 = 0x2C;
    pub const AC_STATUS_MESSAGE: u8 = 0x2D;
}

#[derive(Debug)]
pub enum Message {
    /// Group control messages are to control all groups.
    /// Each message to AirTouch is to control one specific group.
    GroupControl(GroupControlMessage),
    /// The group status for all groups.
    GroupStatuses(Vec<GroupStatus>),
    /// Control a specific AC unit.
    /// Each message to AirTouch is to control one specific AC unit.
    ACControl(ACControlMessage),
    /// The AC status for all AC units.
    ACStatuses(Vec<ACStatus>),
}

#[derive(Debug)]
pub struct GroupControlMessage {
    /// Valid values are `0`-`15`.
    group_number: u8,
    /// The target that the AC should be aiming for; that is, a specific
    /// temperature or percentage value.
    target_value: states::SetGroupSettingValue,
    /// If setting_value is not `Maintain`, `Decrease`, or `Increase`, this
    /// will be ignored.
    control_method: states::SetGroupControlMethod,
    /// Whether or not the group is on/off, or in turbo.
    power: states::SetGroupPower,
}

impl GroupControlMessage {
    pub fn new(
        group_number: u8,
        target_value: states::SetGroupSettingValue,
        control_method: states::SetGroupControlMethod,
        power: states::SetGroupPower,
    ) -> [u8; 14] {
        assert!(group_number <= 15);

        let mut value: u8 = 0;
        let target_value = match target_value {
            states::SetGroupSettingValue::Maintain => 0,
            states::SetGroupSettingValue::Decrease => 0b010,
            states::SetGroupSettingValue::Increase => 0b011,
            states::SetGroupSettingValue::SetOpenPercentage(v) => {
                assert!(v <= 100);
                value = v;
                0b100
            }
            states::SetGroupSettingValue::SetTargetSetpoint(v) => {
                value = v;
                0b101
            }
        };

        let control_method = match control_method {
            states::SetGroupControlMethod::Maintain => 0,
            states::SetGroupControlMethod::Toggle => 0b01,
            states::SetGroupControlMethod::Percentage => 0b10,
            states::SetGroupControlMethod::Temperature => 0b11,
        };

        let power = match power {
            states::SetGroupPower::Maintain => 0,
            states::SetGroupPower::ChangeToNextState => 0b001,
            states::SetGroupPower::SetOff => 0b010,
            states::SetGroupPower::SetOn => 0b011,
            states::SetGroupPower::SetTurbo => 0b101,
        };

        let byte2 = (target_value << 6) | (control_method << 4) | power;

        let crc = crc16::State::<MODBUS>::calculate(&[
            0x80,
            0xb0,
            0x01,
            MessageType::GROUP_CONTROL_MESSAGE,
            0x00,
            0x04,
            group_number,
            byte2,
            value,
            0,
        ]);

        return [
            0x55,
            0x55,
            0x80,
            0xb0,
            0x01,
            MessageType::GROUP_CONTROL_MESSAGE,
            0x00,
            0x04,
            group_number,
            byte2,
            value,
            0,
            (crc >> 8) as u8,
            (crc & 0xff) as u8,
        ];
    }
}

#[derive(Debug)]
pub struct GroupStatus {
    power_state: states::GroupPowerState,
    /// Valid values are `0`-`15`.
    group_number: u8,
    control_method: states::GroupControlMethod,
    open_percentage: f64,
    battery_low: bool,
    turbo_support: bool,
    target_setpoint: f64,
    has_sensor: bool,
    current_temperature: Option<f64>,
    spill: bool,
}

#[derive(Debug)]
pub struct ACControlMessage {
    power: states::SetACPowerState,
    /// Valid values are `0`-`3`.
    number: u8,
    mode: Option<states::ACMode>,
    fan_speed: states::ACFanSpeed,
    setpoint_temp: states::SetACSetpoint,
}

#[derive(Debug)]
pub struct ACStatus {
    power_state: Option<states::ACPowerState>,
    /// Valid values are `0`-`3`.
    number: u8,
    mode: Option<states::ACMode>,
    fan_speed: Option<states::ACFanSpeed>,
    spill: bool,
    has_timer: bool,
    target_setpoint: f64,
    temperature: Option<f64>,
    // TODO: Add error
}

/// Parses a byte array into the relevant message.
pub fn parse_message(bytes: &[u8]) -> Result<Message, &'static str> {
    if bytes.len() < 10 {
        return Err("bytes length is too short to be a valid AT4 message.");
    }

    // Check header bytes
    if bytes[0] != 0x55 || bytes[1] != 0x55 {
        return Err("invalid header.");
    }

    // Check address byte
    if bytes[3] != 0x80 && bytes[3] != 0x90 {
        return Err("invalid address byte.");
    }

    // bytes 7, 8
    let message_length: usize = (((bytes[6] as u16) << 2) | (bytes[7] as u16)) as usize;
    if bytes.len() < (message_length + 10) {
        return Err("bytes length is not long enough for parsed length.");
    }

    // data to be checked by the crc algorithm
    let data = bytes.get(2..(8 + message_length)).unwrap();

    // crc values to check against
    let crc = bytes
        .get((8 + message_length)..(8 + message_length + 2))
        .unwrap();

    let crc: u16 = (crc[0] as u16) << 2 | (crc[1] as u16);

    // validate crc
    if crc16::State::<MODBUS>::calculate(data) != crc {
        return Err("crc check failed.");
    }

    // TODO: Add the message ID to the return messages

    // Check according to message type
    match bytes[5] {
        // Group status
        MessageType::GROUP_STATUS_MESSAGE => todo!(),
        // AC status
        MessageType::AC_STATUS_MESSAGE => todo!(),
        _ => return Err("invalid message type."),
    }
}

#[cfg(test)]
mod tests {
    use super::parse_message;
    use super::GroupControlMessage;

    #[test]
    #[should_panic(expected = "bytes length is too short to be a valid AT4 message.")]
    fn test_parse_error_on_invalid_length() {
        parse_message(&[]).unwrap();
    }

    #[test]
    #[should_panic(expected = "invalid header.")]
    fn test_parse_error_on_invalid_header() {
        parse_message(&[0; 10]).unwrap();
    }

    #[test]
    #[should_panic(expected = "invalid address byte.")]
    fn test_parse_error_on_address_byte() {
        parse_message(&[0x55, 0x55, 0, 0, 0, 0, 0, 0, 0, 0]).unwrap();
    }

    #[test]
    #[should_panic(expected = "bytes length is not long enough for parsed length.")]
    fn test_parse_error_on_invalid_parsed_length() {
        parse_message(&[0x55, 0x55, 0, 0x80, 0, 0, 0xff, 0xab, 0, 0, 0]).unwrap();
    }

    #[test]
    #[should_panic(expected = "crc check failed.")]
    fn test_parse_error_on_invalid_crc() {
        parse_message(&[
            0x55, 0x55, 0xb0, 0x80, 0x01, 0x2b, 0x00, 0x0c, 0x40, 0x64, 0x00, 0x00, 0xff, 0x00,
            0x41, 0xe4, 0x1a, 0x80, 0x61, 0x80, 0x65, 0x78,
        ])
        .unwrap();
    }

    #[test]
    fn test_parses_message() {
        let result = parse_message(&[
            0x55, 0x55, 0xb0, 0x80, 0x01, 0x2b, 0x00, 0x0c, 0x40, 0x64, 0x00, 0x00, 0xff, 0x00,
            0x41, 0xe4, 0x1a, 0x80, 0x61, 0x80, 0x65, 0x79,
        ]);

        match result {
            Err(reason) => println!("error: {}", reason),
            _ => {}
        }
    }

    #[test]
    fn test_create_control_packet() {
        let packet = GroupControlMessage::new(
            1,
            crate::states::SetGroupSettingValue::Maintain,
            crate::states::SetGroupControlMethod::Maintain,
            crate::states::SetGroupPower::SetOff,
        );

        assert_eq!(
            packet,
            [0x55, 0x55, 0x80, 0xb0, 0x1, 0x2a, 0x0, 0x4, 0x1, 0x2, 0x0, 0x0, 0xda, 0x59]
        )
    }
}
