use std::pin::Pin;
use std::sync::{Arc, Weak};
use std::sync::atomic::AtomicBool;
use std::task::{Context, Poll};
use futures::future::{Future, FutureExt};
use futures::lock::Mutex;

#[derive(Debug)]
pub struct AsyncBus<T: Clone> {
    buffer: Arc<Mutex<Vec<T>>>,
}

impl<T: Clone> AsyncBus<T> {
    pub fn new() -> AsyncBus<T> {
        AsyncBus {
            buffer: Arc::new(Mutex::new(Vec::new())),
        }
    }

    pub async fn queue(&self, val: T) {
        self.buffer.lock().await.push(val);
    }

    pub async fn new_rx(self: Arc<AsyncBus<T>>) -> AsyncBusReceiver<T> {
        println!("creating new receiver...");
        AsyncBusReceiver {
            source: Arc::downgrade(&self),
            buffer: Arc::clone(&self.buffer),
            idx: 0
        }
    }
}

pub struct AsyncBusReceiver<T: Clone> {
    source: Weak<AsyncBus<T>>,
    buffer: Arc<Mutex<Vec<T>>>,
    idx: usize,
}

impl<T: Clone> AsyncBusReceiver<T> {
    pub fn source(&self) -> Option<Arc<AsyncBus<T>>> {
        self.source.upgrade()
    }

    pub fn next(&mut self) -> BusFuture<T> {
        let fut = BusFuture {
            closed: self.source().is_none(),
            target_idx: self.idx,
            buffer: Arc::clone(&self.buffer),
        };
        self.idx += 1;
        fut
    }
}

// TODO: Maybe change Mutex<T> to RwLock<T>?
pub struct BusFuture<T: Clone> {
    closed: bool,
    buffer: Arc<Mutex<Vec<T>>>,
    target_idx: usize,
}

impl<T: Clone> Future for BusFuture<T> {
    type Output = Option<T>;

    fn poll(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Self::Output> {
        match Pin::new(&mut self.buffer.lock()).poll(cx) {
            Poll::Pending => Poll::Pending,
            Poll::Ready(buffer) => {
                match buffer.get(self.target_idx) {
                    Some(v) => Poll::Ready(Some(v.clone())),
                    None => if self.closed {
                        Poll::Ready(None)
                    } else {
                        Poll::Pending
                    },
                }
            }
        }
    }
}



#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn it_broadcasts() {
        futures::executor::block_on(async_it_broadcasts());
    }

    async fn async_it_broadcasts() {
        let bus = Arc::new(AsyncBus::<i32>::new());
        let mut rx = Arc::clone(&bus).new_rx().await;
        bus.queue(5).await;
        drop(bus);
        assert_eq!(rx.next().await, Some(5));
        //dbg!(bus);
    }
}
