use std::{
    error::Error,
    future::Future,
    io::SeekFrom,
    path::{Path, PathBuf},
    pin::Pin,
    sync::Arc,
    time::Duration,
};

use tokio::{
    fs::File,
    io::{AsyncReadExt, AsyncSeekExt, BufReader},
    sync::mpsc::{
        error::{SendError, TryRecvError},
        Receiver, Sender,
    },
    time::sleep,
};

#[derive(Debug)]
struct LogBufReader {
    file: BufReader<File>,
    sender: Arc<Sender<Vec<u8>>>,
    path: PathBuf,
    last_ctime: u64,
}

#[derive(Debug)]
pub struct LogWatcher {
    receiver: Receiver<Vec<u8>>,
    sender: Arc<Sender<Vec<u8>>>,
    path: PathBuf,
    signal_tx: Sender<LogWatcherSignal>,
    signal_rx: std::sync::Mutex<Option<Receiver<LogWatcherSignal>>>,
}

#[derive(Debug)]
enum DetachedLogWatcher {
    Initializing(LogBufReader),
    Waiting(LogBufReader),
    Reading(LogBufReader),
    Missing(LogBufReader),
    Reloading((PathBuf, Arc<Sender<Vec<u8>>>)),
    Closed,
}
#[derive(Debug)]
pub enum LogWatcherSignal {
    Close,
    Reload,
    Swap(PathBuf),
}

type BoxedError = Box<dyn Error + 'static + Send + Sync>;
type BoxedResult<T> = Result<T, BoxedError>;
type SpawnFnResult = Pin<Box<dyn Future<Output = BoxedResult<()>> + Send + Sync>>;

impl LogWatcher {
    pub fn new(file_path: impl Into<PathBuf>) -> Self {
        let (sender, receiver) = tokio::sync::mpsc::channel(256);
        let (signal_tx, signal_rx) = tokio::sync::mpsc::channel(256);

        Self {
            receiver,
            sender: Arc::new(sender),
            path: file_path.into(),
            signal_rx: Some(signal_rx).into(),
            signal_tx,
        }
    }

    pub async fn send_signal(
        &self,
        signal: LogWatcherSignal,
    ) -> Result<(), SendError<LogWatcherSignal>> {
        self.signal_tx.send(signal).await
    }

    pub async fn read_message(&mut self) -> Option<Vec<u8>> {
        self.receiver.recv().await
    }

    pub fn try_read_message(&mut self) -> Result<Vec<u8>, TryRecvError> {
        self.receiver.try_recv()
    }

    pub fn spawn(&self, skip_to_end: bool) -> SpawnFnResult {
        let sender = self.sender.clone();
        let path = self.path.clone();

        let signal_rx = self.signal_rx.lock().unwrap().take();

        if signal_rx.is_none() {
            panic!("Log watcher spanwed twice {:?}", self);
        };

        let mut signal_rx = signal_rx.unwrap();

        let future: SpawnFnResult = Box::pin(async move {
            let file = File::open(&path).await?;
            let mut detached = if skip_to_end {
                DetachedLogWatcher::Initializing(LogBufReader {
                    file: BufReader::new(file),
                    sender,
                    path: path.clone(),
                    last_ctime: get_c_time(&path).await.unwrap(),
                })
            } else {
                DetachedLogWatcher::Reading(LogBufReader {
                    file: BufReader::new(file),
                    sender,
                    path: path.clone(),
                    last_ctime: get_c_time(&path).await.unwrap(),
                })
            };

            loop {
                match signal_rx.try_recv() {
                    Ok(LogWatcherSignal::Close) => {
                        detached.close().await;
                    }
                    Ok(LogWatcherSignal::Reload) => {
                        detached.reload().await;
                    }
                    Ok(LogWatcherSignal::Swap(path)) => {
                        detached.swap(path).await;
                    }
                    Err(err) => {
                        if err == TryRecvError::Disconnected {
                            break;
                        }
                    }
                }

                match detached {
                    DetachedLogWatcher::Closed => {
                        break;
                    }
                    _ => {
                        detached = detached
                            .next()
                            .await
                            .expect("failed to move next on detached log watcher");
                    }
                }
            }

            Ok(())
        });
        future
    }
}

impl DetachedLogWatcher {
    pub async fn next(self) -> Result<Self, std::io::Error> {
        match self {
            DetachedLogWatcher::Initializing(mut inner) => {
                inner.skip_file().await?;
                Ok(DetachedLogWatcher::Waiting(inner))
            }
            DetachedLogWatcher::Waiting(mut inner) => match inner.read_next().await {
                Ok(size) if size > 4096 => Ok(DetachedLogWatcher::Reading(inner)),
                Ok(size) => {
                    if size == 0 {
                        let curr_ctime = get_c_time(&inner.path).await?;

                        if curr_ctime > inner.last_ctime {
                            return Ok(DetachedLogWatcher::Missing(inner));
                        }

                        sleep(Duration::from_secs(1)).await;
                        Ok(DetachedLogWatcher::Waiting(inner))
                    } else {
                        sleep(Duration::from_secs(1)).await;
                        Ok(DetachedLogWatcher::Waiting(inner))
                    }
                }
                Err(err) => match err.kind() {
                    std::io::ErrorKind::NotFound => Ok(DetachedLogWatcher::Missing(inner)),
                    _ => Err(err),
                },
            },
            DetachedLogWatcher::Reading(mut inner) => match inner.read_next().await {
                Ok(size) if size < 4096 => Ok(DetachedLogWatcher::Waiting(inner)),
                Ok(_) => Ok(DetachedLogWatcher::Reading(inner)),
                Err(err) => match err.kind() {
                    std::io::ErrorKind::NotFound => Ok(DetachedLogWatcher::Missing(inner)),
                    _ => Err(err),
                },
            },
            DetachedLogWatcher::Missing(inner) => {
                inner.sender.try_send(inner.file.buffer().to_vec()).ok();
                Ok(DetachedLogWatcher::Reloading((inner.path, inner.sender)))
            }
            DetachedLogWatcher::Reloading((path, sender)) => {
                let file_exists = match tokio::fs::metadata(&path).await {
                    Ok(meta) => Ok(meta.is_file()),
                    Err(err) => match err.kind() {
                        std::io::ErrorKind::NotFound | std::io::ErrorKind::PermissionDenied => {
                            Ok(false)
                        }
                        _ => Err(err),
                    },
                }?;

                if file_exists {
                    let new_inner = LogBufReader {
                        file: BufReader::new(File::open(&path).await?),
                        path: path.clone(),
                        sender,
                        last_ctime: get_c_time(&path).await.unwrap(),
                    };

                    Ok(DetachedLogWatcher::Waiting(new_inner))
                } else {
                    sleep(Duration::from_secs(1)).await;
                    Ok(DetachedLogWatcher::Reloading((path, sender)))
                }
            }
            DetachedLogWatcher::Closed => Ok(DetachedLogWatcher::Closed),
        }
    }

    pub async fn close(&mut self) {
        match self {
            DetachedLogWatcher::Initializing(inner)
            | DetachedLogWatcher::Waiting(inner)
            | DetachedLogWatcher::Reading(inner)
            | DetachedLogWatcher::Missing(inner) => {
                inner.read_next().await.ok();
                *self = DetachedLogWatcher::Closed
            }
            DetachedLogWatcher::Reloading(_) => *self = DetachedLogWatcher::Closed,
            DetachedLogWatcher::Closed => {}
        }
    }

    pub async fn reload(&mut self) {
        match self {
            DetachedLogWatcher::Initializing(inner)
            | DetachedLogWatcher::Waiting(inner)
            | DetachedLogWatcher::Reading(inner)
            | DetachedLogWatcher::Missing(inner) => {
                let result = inner.read_next().await.unwrap_or(0);

                if result == 0 {
                    inner.sender.try_send(inner.file.buffer().to_vec()).ok();
                }
                *self = DetachedLogWatcher::Reloading((inner.path.clone(), inner.sender.clone()));
            }
            DetachedLogWatcher::Reloading(_) | DetachedLogWatcher::Closed => {}
        }
    }

    pub async fn swap(&mut self, path: PathBuf) {
        match self {
            DetachedLogWatcher::Initializing(inner)
            | DetachedLogWatcher::Waiting(inner)
            | DetachedLogWatcher::Reading(inner)
            | DetachedLogWatcher::Missing(inner) => {
                let result = inner.read_next().await.unwrap_or(0);

                if result == 0 {
                    inner.sender.try_send(inner.file.buffer().to_vec()).ok();
                }
                *self = DetachedLogWatcher::Reloading((path, inner.sender.clone()));
            }
            DetachedLogWatcher::Reloading((_old_path, sender)) => {
                *self = DetachedLogWatcher::Reloading((path, sender.clone()));
            }
            DetachedLogWatcher::Closed => {}
        }
    }
}

impl LogBufReader {
    async fn read_next(&mut self) -> Result<usize, std::io::Error> {
        let mut buffer: Vec<u8> = Vec::new();
        let result: Result<usize, std::io::Error> = self.file.read_to_end(&mut buffer).await;

        match result {
            Ok(size) if size > 0 => match self.sender.try_send(buffer) {
                Ok(_) => {
                    self.last_ctime = get_c_time(&self.path).await?;
                    Ok(size)
                }
                Err(_) => Err(std::io::Error::new(
                    std::io::ErrorKind::NotConnected,
                    "failed to send to channel",
                )),
            },
            Ok(size) => Ok(size),
            Err(err) => match err.kind() {
                std::io::ErrorKind::UnexpectedEof => Ok(0),
                _ => Err(err),
            },
        }
    }

    async fn skip_file(&mut self) -> Result<(), std::io::Error> {
        self.file.seek(SeekFrom::End(0)).await?;
        Ok(())
    }
}

#[cfg(windows)]
async fn get_c_time(path: &Path) -> Result<u64, std::io::Error> {
    use std::os::windows::prelude::MetadataExt;

    let meta = tokio::fs::metadata(path).await?;
    Ok(meta.last_write_time())
}

#[cfg(unix)]
async fn get_c_time(path: &Path) -> Result<u64, std::io::Error> {
    use std::os::unix::prelude::MetadataExt;

    let meta = tokio::fs::metadata(path).await?;
    Ok(meta.ctime() as u64)
}
