use axum::body::{boxed, Body, BoxBody};
use axum::http;
use axum::http::header::HeaderName;
use axum::http::{HeaderValue, Response};
use axum::response::IntoResponse;

/// Responder for a GraphQL response.
///
/// This contains a batch response, but since regular responses are a type of batch response it
/// works for both.
pub struct GraphQLResponse(pub async_graphql::BatchResponse);

impl From<async_graphql::Response> for GraphQLResponse {
    fn from(resp: async_graphql::Response) -> Self {
        Self(resp.into())
    }
}

impl From<async_graphql::BatchResponse> for GraphQLResponse {
    fn from(resp: async_graphql::BatchResponse) -> Self {
        Self(resp)
    }
}

impl IntoResponse for GraphQLResponse {
    fn into_response(self) -> Response<BoxBody> {
        let body: Body = serde_json::to_string(&self.0).unwrap().into();
        let mut resp = Response::new(boxed(body));
        resp.headers_mut().insert(
            http::header::CONTENT_TYPE,
            HeaderValue::from_static("application/json"),
        );
        if self.0.is_ok() {
            if let Some(cache_control) = self.0.cache_control().value() {
                if let Ok(value) = HeaderValue::from_str(&cache_control) {
                    resp.headers_mut()
                        .insert(http::header::CACHE_CONTROL, value);
                }
            }
        }
        for (name, value) in self.0.http_headers() {
            if let (Ok(name), Ok(value)) = (
                HeaderName::try_from(name.as_bytes()),
                HeaderValue::from_str(value),
            ) {
                resp.headers_mut().insert(name, value);
            }
        }
        resp
    }
}
