// Copyright 2021-2022 System76 <info@system76.com>
// SPDX-License-Identifier: MPL-2.0

//! Asynchronously fetch files from HTTP servers
//!
//! - Concurrently fetch multiple files at the same time.
//! - Define alternative mirrors for the source of a file.
//! - Use multiple concurrent connections per file.
//! - Use mirrors for concurrent connections.
//! - Resume a download which has been interrupted.
//! - Progress events for fetches
//!
//! ```
//! let results_stream = Fetcher::default()
//!     // Define a max number of ranged connections per file.
//!     .connections_per_file(4)
//!     // Max size of a connection's part, concatenated on completion.
//!     .max_part_size(4 * 1024 * 1024)
//!     // An `Arc<AtomicBool>` which can be used to interrupt the download.
//!     .cancel(cancellable)
//!     // The channel for sending progress notifications.
//!     .events(events_tx)
//!     // Maximum number of retry attempts.
//!     .retries(3)
//!     // How long to wait before aborting a download that hasn't progressed.
//!     .timeout(Duration::from_secs(15))
//!     // Finalize the struct into an `Arc` for use with fetching.
//!     .build()
//!     // Take a stream of `Source` inputs and generate a stream of fetches.
//!     .requests_stream(input_stream)
//!     // Fetches up to 8 sources concurrently
//!     .buffered(8);
//! ```

#[macro_use]
extern crate derive_new;
#[macro_use]
extern crate derive_setters;
#[macro_use]
extern crate log;
#[macro_use]
extern crate thiserror;

pub mod checksum;
mod checksum_system;
mod concatenator;
mod get;
mod get_many;
mod range;
mod source;
mod time;

pub use self::checksum_system::*;
pub use self::concatenator::*;
pub use self::source::*;

use self::get::{get, FetchLocation};
use self::get_many::get_many;
use self::time::{date_as_timestamp, update_modified};

use futures::{
    prelude::*,
    stream::{self, StreamExt},
};
use http::StatusCode;
use httpdate::HttpDate;
use isahc::config::Configurable;
use isahc::{AsyncBody, HttpClient as Client, Request, Response};
use numtoa::NumToA;
use std::{
    fmt::Debug,
    future::Future,
    io,
    path::Path,
    sync::{
        atomic::{AtomicBool, Ordering},
        Arc,
    },
    time::{Duration, UNIX_EPOCH},
};
use tokio::fs;
use tokio::sync::mpsc;

pub type EventSender<Data> = mpsc::UnboundedSender<(Arc<Path>, Data, FetchEvent)>;
pub type Output<T> = (Arc<Path>, Result<T, Error>);

/// An error from the asynchronous file fetcher.
#[derive(Debug, Error)]
pub enum Error {
    #[error("task was cancelled")]
    Cancelled,
    #[error("http client error")]
    Client(isahc::Error),
    #[error("unable to concatenate fetched parts")]
    Concatenate(#[source] io::Error),
    #[error("unable to create file")]
    FileCreate(#[source] io::Error),
    #[error("unable to set timestamp on {:?}", _0)]
    FileTime(Arc<Path>, #[source] io::Error),
    #[error("content length is an invalid range")]
    InvalidRange(#[source] io::Error),
    #[error("unable to remove file with bad metadata")]
    MetadataRemove(#[source] io::Error),
    #[error("destination has no file name")]
    Nameless,
    #[error("unable to open fetched part")]
    OpenPart(Arc<Path>, #[source] io::Error),
    #[error("destination lacks parent")]
    Parentless,
    #[error("connection timed out")]
    TimedOut,
    #[error("error writing to file")]
    Write(#[source] io::Error),
    #[error("failed to rename partial to destination")]
    Rename(#[source] io::Error),
    #[error("server responded with an error: {}", _0)]
    Status(StatusCode),
    #[error("internal tokio join handle error")]
    TokioSpawn(#[source] tokio::task::JoinError),
}

impl From<isahc::Error> for Error {
    fn from(e: isahc::Error) -> Self {
        Self::Client(e)
    }
}

/// Events which are submitted by the fetcher.
#[derive(Debug)]
pub enum FetchEvent {
    /// Signals that this file was already fetched.
    AlreadyFetched,
    /// States that we know the length of the file being fetched.
    ContentLength(u64),
    /// Notifies that the file has been fetched.
    Fetched,
    /// Notifies that a file is being fetched.
    Fetching,
    /// Reports the amount of bytes that have been read for a file.
    Progress(u64),
    /// Reports that a part of a file is being fetched.
    PartFetching(u64),
    /// Reports that a part has been fetched.
    PartFetched(u64),
    /// Notification that a fetch is being re-attempted.
    Retrying,
}

/// An asynchronous file fetcher for clients fetching files.
///
/// The futures generated by the fetcher are compatible with single and multi-threaded
/// runtimes, allowing you to choose between the runtime that works best for your
/// application. A single-threaded runtime is generally recommended for fetching files,
/// as your network connection is unlikely to be faster than a single CPU core.
#[derive(new, Setters)]
pub struct Fetcher<Data> {
    #[setters(skip)]
    client: Client,

    /// When set, cancels any active operations.
    #[new(default)]
    #[setters(strip_option)]
    cancel: Option<Arc<AtomicBool>>,

    /// The number of concurrent connections to sustain per file being fetched.
    /// # Note
    /// Defaults to 1 connection
    #[new(value = "1")]
    connections_per_file: u16,

    /// The number of attempts to make when a request fails.
    /// # Note
    /// Defaults to 3 retries.
    #[new(value = "3")]
    retries: u16,

    /// The maximum size of a part file when downloading in parts.
    /// # Note
    /// Defaults to 2 MiB.
    #[new(value = "2 * 1024 * 1024")]
    max_part_size: u32,

    /// The time to wait between chunks before giving up.
    #[new(default)]
    #[setters(strip_option)]
    timeout: Option<Duration>,

    /// Holds a sender for submitting events to.
    #[new(default)]
    #[setters(into)]
    #[setters(strip_option)]
    events: Option<Arc<EventSender<Arc<Data>>>>,
}

impl<Data> Default for Fetcher<Data> {
    fn default() -> Self {
        Self::new(
            Client::builder()
                .low_speed_timeout(1, std::time::Duration::from_secs(15))
                .redirect_policy(isahc::config::RedirectPolicy::Follow)
                .build()
                .expect("failed to build HTTP client"),
        )
    }
}

impl<Data: Send + Sync + 'static> Fetcher<Data> {
    /// Finalizes the fetcher to prepare it for fetch tasks.
    pub fn build(self) -> Arc<Self> {
        Arc::new(self)
    }

    /// Build a stream that will perform fetches when polled.
    pub fn requests_stream(
        self: Arc<Self>,
        inputs: impl Stream<Item = (Source, Arc<Data>)> + Unpin + Send + 'static,
    ) -> impl Stream<
        Item = impl Future<Output = (Arc<Path>, Arc<Data>, Result<(), Error>)> + Send + 'static,
    > + Send
           + Unpin
           + 'static {
        inputs.map(move |(source, extra)| {
            let fetcher = self.clone();

            async move {
                let Source { dest, urls, part } = source;

                let result = match part {
                    Some(part) => match fetcher
                        .clone()
                        .request(urls, part.clone(), extra.clone())
                        .await
                    {
                        Ok(()) => fs::rename(&*part, &*dest).await.map_err(Error::Rename),
                        Err(why) => Err(why),
                    },
                    None => {
                        fetcher
                            .clone()
                            .request(urls, dest.clone(), extra.clone())
                            .await
                    }
                };

                (dest, extra, result)
            }
        })
    }

    /// Request a file from one or more URIs.
    ///
    /// At least one URI must be provided as a source for the file. Each additional URI
    /// serves as a mirror for failover and load-balancing purposes.
    pub async fn request(
        self: Arc<Self>,
        uris: Arc<[Box<str>]>,
        to: Arc<Path>,
        extra: Arc<Data>,
    ) -> Result<(), Error> {
        self.send(|| (to.clone(), extra.clone(), FetchEvent::Fetching));

        remove_parts(&to).await;

        let fetch = || async {
            self.clone()
                .inner_request(uris.clone(), to.clone(), extra.clone())
                .await
        };

        let cleanup = || async {
            remove_parts(&to).await;

            self.send(|| (to.clone(), extra.clone(), FetchEvent::Fetched));
        };

        let mut attempts = 0;

        let result = match fetch().await {
            Ok(()) => Ok(()),
            Err(mut why) => {
                while attempts < self.retries {
                    attempts += 1;

                    if self.cancelled() {
                        cleanup().await;
                        return Err(Error::Cancelled);
                    }

                    self.send(|| (to.clone(), extra.clone(), FetchEvent::Retrying));

                    match fetch().await {
                        Ok(()) => {
                            cleanup().await;
                            return Ok(());
                        }

                        Err(cause) => why = cause,
                    }
                }

                Err(why)
            }
        };

        cleanup().await;

        result
    }

    async fn inner_request(
        self: Arc<Self>,
        uris: Arc<[Box<str>]>,
        to: Arc<Path>,
        extra: Arc<Data>,
    ) -> Result<(), Error> {
        let mut length = None;
        let mut modified = None;
        let mut resume = 0;

        let head_response = head(&self.client, &*uris[0]).await?;

        if let Some(response) = head_response.as_ref() {
            length = response.content_length();
            modified = response.last_modified();
        }

        // If the file already exists, validate that it is the same.
        if to.exists() {
            if let (Some(length), Some(last_modified)) = (length, modified) {
                match fs::metadata(to.as_ref()).await {
                    Ok(metadata) => {
                        let modified = metadata.modified().map_err(Error::Write)?;
                        let ts = modified
                            .duration_since(UNIX_EPOCH)
                            .expect("time went backwards");

                        if metadata.len() == length {
                            if ts.as_secs() == date_as_timestamp(last_modified) {
                                self.send(|| (to, extra.clone(), FetchEvent::AlreadyFetched));
                                return Ok(());
                            } else {
                                error!("removing file with outdated timestamp: {:?}", to);
                                let _ = fs::remove_file(to.as_ref())
                                    .await
                                    .map_err(Error::MetadataRemove)?;
                            }
                        } else {
                            resume = metadata.len();
                        }
                    }
                    Err(why) => {
                        error!("failed to fetch metadata of {:?}: {}", to, why);
                        fs::remove_file(to.as_ref())
                            .await
                            .map_err(Error::MetadataRemove)?;
                    }
                }
            }
        }

        // If set, this will use multiple connections to download a file in parts.
        if self.connections_per_file > 1 {
            if let Some(length) = length {
                if supports_range(&self.client, &*uris[0], resume, Some(length)).await? {
                    self.send(|| (to.clone(), extra.clone(), FetchEvent::ContentLength(length)));

                    if resume != 0 {
                        self.send(|| (to.clone(), extra.clone(), FetchEvent::Progress(resume)));
                    }

                    get_many(
                        self.clone(),
                        to.clone(),
                        uris,
                        resume,
                        length,
                        modified,
                        extra,
                    )
                    .await?;

                    if let Some(modified) = modified {
                        update_modified(&to, modified)?;
                    }

                    return Ok(());
                }
            }
        }

        if let Some(length) = length {
            self.send(|| (to.clone(), extra.clone(), FetchEvent::ContentLength(length)));

            if resume > length {
                resume = 0;
            }
        }

        let mut request = Request::get(&*uris[0]);

        if resume != 0 {
            if let Ok(true) = supports_range(&self.client, &*uris[0], resume, length).await {
                request = request.header("Range", range::to_string(resume, length));
            } else {
                resume = 0;
            }
        }

        let path = match crate::get(
            self.clone(),
            request,
            FetchLocation::create(to.clone(), length, resume != 0).await?,
            to.clone(),
            &mut modified,
            extra.clone(),
        )
        .await
        {
            Ok(path) => path,

            Err(Error::Status(StatusCode::NOT_MODIFIED)) => to,

            // Server does not support if-modified-since
            Err(Error::Status(StatusCode::NOT_IMPLEMENTED)) => {
                let request = Request::get(&*uris[0]);
                crate::get(
                    self.clone(),
                    request,
                    FetchLocation::create(to.clone(), length, resume != 0).await?,
                    to.clone(),
                    &mut modified,
                    extra.clone(),
                )
                .await?
            }

            Err(why) => return Err(why),
        };

        if let Some(modified) = modified {
            update_modified(&path, modified)?;
        }

        Ok(())
    }

    fn cancelled(&self) -> bool {
        self.cancel
            .as_ref()
            .map_or(false, |cancel| cancel.load(Ordering::SeqCst))
    }

    fn send(&self, event: impl FnOnce() -> (Arc<Path>, Arc<Data>, FetchEvent)) {
        if let Some(sender) = self.events.as_ref() {
            let _ = sender.send(event());
        }
    }
}

pub async fn head(client: &Client, uri: &str) -> Result<Option<Response<AsyncBody>>, Error> {
    let request = Request::head(uri).body(()).unwrap();

    match validate(client.send_async(request).await?).map(Some) {
        result @ Ok(_) => result,
        Err(Error::Status(StatusCode::NOT_MODIFIED))
        | Err(Error::Status(StatusCode::NOT_IMPLEMENTED)) => Ok(None),
        Err(other) => Err(other),
    }
}

pub async fn supports_range(
    client: &Client,
    uri: &str,
    resume: u64,
    length: Option<u64>,
) -> Result<bool, Error> {
    let request = Request::head(uri)
        .header("Range", range::to_string(resume, length).as_str())
        .body(())
        .unwrap();

    let response = client.send_async(request).await?;

    if response.status() == StatusCode::PARTIAL_CONTENT {
        if let Some(header) = response.headers().get("Content-Range") {
            if let Ok(header) = header.to_str() {
                if header.starts_with(&format!("bytes {}-", resume)) {
                    return Ok(true);
                }
            }
        }

        Ok(false)
    } else {
        validate(response).map(|_| false)
    }
}

pub async fn timed<F, T>(duration: Duration, future: F) -> Result<T, Error>
where
    F: Future<Output = T> + Unpin,
{
    let timeout = async move {
        tokio::time::sleep(duration).await;
        Err(Error::TimedOut)
    };

    let result = async move { Ok(future.await) };

    futures::pin_mut!(timeout);
    futures::pin_mut!(result);

    futures::future::select(timeout, result)
        .await
        .factor_first()
        .0
}

fn validate(response: Response<AsyncBody>) -> Result<Response<AsyncBody>, Error> {
    let status = response.status();

    if status.is_informational() || status.is_success() {
        Ok(response)
    } else {
        Err(Error::Status(status))
    }
}

trait ResponseExt {
    fn content_length(&self) -> Option<u64>;
    fn last_modified(&self) -> Option<HttpDate>;
}

impl ResponseExt for Response<AsyncBody> {
    fn content_length(&self) -> Option<u64> {
        let header = self.headers().get("content-length")?;
        header.to_str().ok()?.parse::<u64>().ok()
    }

    fn last_modified(&self) -> Option<HttpDate> {
        let header = self.headers().get("last-modified")?;
        httpdate::parse_http_date(header.to_str().ok()?)
            .ok()
            .map(HttpDate::from)
    }
}

/// Cleans up after a process that may have been aborted.
async fn remove_parts(to: &Path) {
    let original_filename = match to.file_name().and_then(|x| x.to_str()) {
        Some(name) => name,
        None => return,
    };

    if let Some(parent) = to.parent() {
        if let Ok(mut dir) = tokio::fs::read_dir(parent).await {
            while let Ok(Some(entry)) = dir.next_entry().await {
                if let Some(entry_name) = entry.file_name().to_str() {
                    if let Some(potential_part) = entry_name.strip_prefix(original_filename) {
                        if potential_part.starts_with(".part") {
                            let path = entry.path();
                            let _ = tokio::fs::remove_file(path).await;
                        }
                    }
                }
            }
        }
    }
}
