//! A simple IRC crate written in rust
//! ```no_run
//! use circe::{commands::Command, Client, Config};
//! fn main() -> Result<(), tokio::io::Error> {
//!     let config = Default::default();
//!     let mut client = Client::new(config).await.unwrap();
//!     client.identify().await.unwrap();
//!
//!     loop {
//!         if let Ok(ref command) = client.read().await {
//!             if let Command::OTHER(line) = command {
//!                 print!("{}", line);
//!             }
//!             if let Command::PRIVMSG(nick, channel, message) = command {
//!                println!("PRIVMSG received from {}: {} {}", nick, channel, message);
//!             }
//!         }
//!         # break;
//!     }
//!     # Ok(())
//! }

#![warn(missing_docs)]
#![allow(clippy::too_many_lines)]
#![cfg_attr(docsrs, feature(doc_cfg))]

#[cfg(feature = "tls")]
use async_native_tls::TlsConnector;

use std::borrow::Cow;
use tokio::io::{AsyncReadExt, AsyncWriteExt, Error};
use tokio::net::TcpStream;

#[cfg(feature = "toml_support")]
use serde_derive::Deserialize;
#[cfg(feature = "toml_support")]
use std::fs::File;
#[cfg(feature = "toml_support")]
use std::path::Path;

/// IRC comamnds
pub mod commands;

/// An IRC client
pub struct Client {
    config: Config,
    #[cfg(not(feature = "tls"))]
    stream: TcpStream,
    #[cfg(feature = "tls")]
    stream: async_native_tls::TlsStream<TcpStream>,
}

/// Config for the IRC client
#[derive(Clone, Debug, Default)]
#[cfg_attr(feature = "toml_support", derive(Deserialize))]
pub struct Config {
    channels: Vec<String>,
    host: String,
    mode: Option<String>,
    nickname: Option<String>,
    port: u16,
    username: String,
}

/// Custom Error for the `read` function
#[derive(Debug)]
pub struct NoNewLines;

impl std::fmt::Display for NoNewLines {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "Now new lines from the stream.")
    }
}

impl std::error::Error for NoNewLines {}

impl Client {
    /// Creates a new client with a given [`Config`].
    /// ```no_run
    /// # use circe::*;
    /// # let config = Default::default();
    /// let mut client = Client::new(config).await.unwrap()?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not connect to the host.
    /// # Panics
    /// Panics if the client can't connect to the given host.
    pub async fn new(config: Config) -> Result<Self, Error> {
        let stream = TcpStream::connect(format!("{}:{}", config.host, config.port))
            .await
            .unwrap();

        #[cfg(feature = "tls")]
        let sslstream: async_native_tls::TlsStream<TcpStream>;

        #[cfg(feature = "tls")]
        {
            let connector = TlsConnector::new();
            sslstream = connector
                .connect(config.host.as_str(), stream)
                .await
                .unwrap();
        };

        #[cfg(feature = "tls")]
        return Ok(Self {
            config,
            stream: sslstream,
        });

        #[cfg(not(feature = "tls"))]
        return Ok(Self { config, stream });
    }

    /// Identify user and joins the in the [`Config`] specified channels.
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default())?;
    /// client.identify()?.await.unwrap();
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn identify(&mut self) -> Result<(), Error> {
        self.write_command(commands::Command::CAP(commands::CapMode::LS))
            .await?;
        self.write_command(commands::Command::CAP(commands::CapMode::END))
            .await?;

        self.write_command(commands::Command::USER(
            self.config.username.clone(),
            "*".into(),
            "*".into(),
            self.config.username.clone(),
        ))
        .await?;

        if let Some(nick) = self.config.nickname.clone() {
            self.write_command(commands::Command::NICK(nick)).await?;
        } else {
            self.write_command(commands::Command::NICK(self.config.username.clone()))
                .await?;
        }

        loop {
            if let Ok(ref command) = self.read().await {
                match command {
                    commands::Command::PING(code) => {
                        self.write_command(commands::Command::PONG(code.to_string()))
                            .await?;
                    }
                    commands::Command::OTHER(line) => {
                        tracing::debug!("{}", line);
                        if line.contains("001") {
                            break;
                        }
                    }
                    _ => {}
                }
            }
        }

        let config = self.config.clone();
        let user = {
            if let Some(nick) = config.nickname {
                nick
            } else {
                config.username
            }
        };
        self.write_command(commands::Command::MODE(user, config.mode))
            .await?;
        for channel in &config.channels {
            self.write_command(commands::Command::JOIN(channel.to_string()))
                .await?;
        }

        tracing::info!("circe has identified");
        Ok(())
    }

    async fn read_string(&mut self) -> Option<String> {
        let mut buffer = [0u8; 512];

        let num_bytes = match self.stream.read(&mut buffer).await {
            Ok(b) => b,
            Err(_) => 0,
        };

        if num_bytes == 0 {
            return None;
        }

        // Slice to the number of bytes that were actually read
        let res = String::from_utf8_lossy(&buffer[..num_bytes]);

        Some(res.into())
    }

    /// Read data coming from the IRC as a [`commands::Command`].
    /// ```no_run
    /// # use circe::*;
    /// # use circe::commands::Command;
    /// # fn main() -> Result<(), tokio::io::Error> {
    /// # let config = Default::default();
    /// # let mut client = Client::new(config)?;
    /// if let Ok(ref command) = client.read().await {
    ///     if let Command::OTHER(line) = command {
    ///         print!("{}", line);
    ///     }
    /// }
    /// # Ok::<(), tokio::io::Error>(())
    /// # }
    /// ```
    /// # Errors
    /// Returns error if there are no new messages. This should not be taken as an actual error, because nothing went wrong.
    pub async fn read(&mut self) -> Result<commands::Command, NoNewLines> {
        if let Some(string) = self.read_string().await {
            let command = commands::Command::command_from_str(&string).await;

            if let commands::Command::PONG(command) = command {
                if let Err(_e) = self.write_command(commands::Command::PONG(command)).await {
                    return Err(NoNewLines);
                }
                return Ok(commands::Command::PONG("".to_string()));
            }

            return Ok(command);
        }

        Err(NoNewLines)
    }

    async fn write(&mut self, data: &str) -> Result<(), Error> {
        let formatted = {
            let new = format!("{}\r\n", data);
            Cow::Owned(new) as Cow<str>
        };

        tracing::debug!("{:?}", formatted);

        self.stream.write_all(formatted.as_bytes()).await?;

        Ok(())
    }

    /// Send a [`commands::Command`] to the IRC.<br>
    /// Not reccomended to use, use the helper functions instead.
    /// ```no_run
    /// # use circe::*;
    /// # use circe::commands::Command;
    /// # let mut client = Client::new(Default::default())?;
    /// client.write_command(Command::PRIVMSG("".to_string(), "#main".to_string(), "Hello".to_string())).await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn write_command(&mut self, command: commands::Command) -> Result<(), Error> {
        use commands::Command::{
            ADMIN, AWAY, CAP, INVITE, JOIN, LIST, MODE, NAMES, NICK, OPER, OTHER, PART, PASS, PING,
            PONG, PRIVMSG, QUIT, TOPIC, USER,
        };
        let computed = match command {
            ADMIN(target) => {
                let formatted = format!("ADMIN {}", target);
                Cow::Owned(formatted) as Cow<str>
            }
            AWAY(message) => {
                let formatted = format!("AWAY {}", message);
                Cow::Owned(formatted) as Cow<str>
            }
            CAP(mode) => {
                use commands::CapMode::{END, LS};
                Cow::Borrowed(match mode {
                    LS => "CAP LS 302",
                    END => "CAP END",
                }) as Cow<str>
            }
            INVITE(username, channel) => {
                let formatted = format!("INVITE {} {}", username, channel);
                Cow::Owned(formatted) as Cow<str>
            }
            JOIN(channel) => {
                let formatted = format!("JOIN {}", channel);
                Cow::Owned(formatted) as Cow<str>
            }
            LIST(channel, server) => {
                let mut formatted = "LIST".to_string();
                if let Some(channel) = channel {
                    formatted.push_str(format!(" {}", channel).as_str());
                }
                if let Some(server) = server {
                    formatted.push_str(format!(" {}", server).as_str());
                }
                Cow::Owned(formatted) as Cow<str>
            }
            NAMES(channel, server) => {
                let formatted = {
                    if let Some(server) = server {
                        format!("NAMES {} {}", channel, server)
                    } else {
                        format!("NAMES {}", channel)
                    }
                };
                Cow::Owned(formatted) as Cow<str>
            }
            NICK(nickname) => {
                let formatted = format!("NICK {}", nickname);
                Cow::Owned(formatted) as Cow<str>
            }
            MODE(target, mode) => {
                let formatted = {
                    if let Some(mode) = mode {
                        format!("MODE {} {}", target, mode)
                    } else {
                        format!("MODE {}", target)
                    }
                };
                Cow::Owned(formatted) as Cow<str>
            }
            OPER(nick, password) => {
                let formatted = format!("OPER {} {}", nick, password);
                Cow::Owned(formatted) as Cow<str>
            }
            OTHER(_) => {
                return Err(Error::new(
                    tokio::io::ErrorKind::Other,
                    "Cannot write commands of type OTHER",
                ));
            }
            PART(target) => {
                let formatted = format!("PART {}", target);
                Cow::Owned(formatted) as Cow<str>
            }
            PASS(password) => {
                let formatted = format!("PASS {}", password);
                Cow::Owned(formatted) as Cow<str>
            }
            PING(target) => {
                let formatted = format!("PING {}", target);
                Cow::Owned(formatted) as Cow<str>
            }
            PONG(code) => {
                let formatted = format!("PONG {}", code);
                Cow::Owned(formatted) as Cow<str>
            }
            PRIVMSG(_, target, message) => {
                let formatted = format!("PRIVMSG {} {}", target, message);
                Cow::Owned(formatted) as Cow<str>
            }
            QUIT(message) => {
                let formatted = format!("QUIT :{}", message);
                Cow::Owned(formatted) as Cow<str>
            }
            TOPIC(channel, topic) => {
                let formatted = {
                    if let Some(topic) = topic {
                        format!("TOPIC {} :{}", channel, topic)
                    } else {
                        format!("TOPIC {}", channel)
                    }
                };
                Cow::Owned(formatted) as Cow<str>
            }
            USER(username, s1, s2, realname) => {
                let formatted = format!("USER {} {} {} :{}", username, s1, s2, realname);
                Cow::Owned(formatted) as Cow<str>
            }
        };

        self.write(&computed).await?;
        Ok(())
    }

    // Helper commands

    /// Helper function for requesting information about the ADMIN of an IRC server
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.admin("192.168.178.100").await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn admin(&mut self, target: &str) -> Result<(), Error> {
        self.write_command(commands::Command::ADMIN(target.to_string()))
            .await?;
        Ok(())
    }

    /// Helper function for setting the users status to AWAY
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.away("AFK").await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn away(&mut self, message: &str) -> Result<(), Error> {
        self.write_command(commands::Command::AWAY(message.to_string()))
            .await?;
        Ok(())
    }

    /// Helper function for sending PRIVMSGs.
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.privmsg("#main", "Hello").await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn privmsg(&mut self, channel: &str, message: &str) -> Result<(), Error> {
        self.write_command(commands::Command::PRIVMSG(
            String::from(""),
            channel.to_string(),
            message.to_string(),
        ))
        .await?;
        Ok(())
    }

    /// Helper function to INVITE people to a channels
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.invite("liblemonirc", "#circe").await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn invite(&mut self, username: &str, channel: &str) -> Result<(), Error> {
        self.write_command(commands::Command::INVITE(
            username.to_string(),
            channel.to_string(),
        ))
        .await?;
        Ok(())
    }

    /// Helper function for sending JOINs.
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.join("#main").await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn join(&mut self, channel: &str) -> Result<(), Error> {
        self.write_command(commands::Command::JOIN(channel.to_string()))
            .await?;
        Ok(())
    }

    /// Helper function for ``LISTing`` channels and modes
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.list(None, None).await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn list(&mut self, channel: Option<&str>, server: Option<&str>) -> Result<(), Error> {
        let channel_config = channel.map(std::string::ToString::to_string);
        let server_config = server.map(std::string::ToString::to_string);
        self.write_command(commands::Command::LIST(channel_config, server_config))
            .await?;
        Ok(())
    }

    /// Helper function for getting all nicknames in a channel
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.names("#main,#circe", None).await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn names(&mut self, channel: &str, server: Option<&str>) -> Result<(), Error> {
        if let Some(server) = server {
            self.write_command(commands::Command::NAMES(
                channel.to_string(),
                Some(server.to_string()),
            ))
            .await?;
        } else {
            self.write_command(commands::Command::NAMES(channel.to_string(), None))
                .await?;
        }
        Ok(())
    }

    /// Helper function to try to register as a channel operator
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.oper("username", "password").await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn oper(&mut self, username: &str, password: &str) -> Result<(), Error> {
        self.write_command(commands::Command::OPER(
            username.to_string(),
            password.to_string(),
        ))
        .await?;
        Ok(())
    }

    /// Helper function for sending MODEs.
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.mode("test", Some("+B")).await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn mode(&mut self, target: &str, mode: Option<&str>) -> Result<(), Error> {
        if let Some(mode) = mode {
            self.write_command(commands::Command::MODE(
                target.to_string(),
                Some(mode.to_string()),
            ))
            .await?;
        } else {
            self.write_command(commands::Command::MODE(target.to_string(), None))
                .await?;
        }
        Ok(())
    }

    /// Helper function for leaving channels.
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.part("#main").await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn part(&mut self, target: &str) -> Result<(), Error> {
        self.write_command(commands::Command::PART(target.to_string()))
            .await?;
        Ok(())
    }

    /// Helper function for setting or getting the topic of a channel
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.topic("#main", Some("main channel")).await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn topic(&mut self, channel: &str, topic: Option<&str>) -> Result<(), Error> {
        if let Some(topic) = topic {
            self.write_command(commands::Command::TOPIC(
                channel.to_string(),
                Some(topic.to_string()),
            ))
            .await?;
        } else {
            self.write_command(commands::Command::TOPIC(channel.to_string(), None))
                .await?;
        }
        Ok(())
    }

    /// Helper function for leaving the IRC server and shutting down the TCP stream afterwards.
    /// ```no_run
    /// # use circe::*;
    /// # let mut client = Client::new(Default::default()).await?;
    /// client.quit(None).await?;
    /// # Ok::<(), tokio::io::Error>(())
    /// ```
    /// # Errors
    /// Returns error if the client could not write to the stream.
    pub async fn quit(&mut self, message: Option<&str>) -> Result<(), Error> {
        if let Some(message) = message {
            self.write_command(commands::Command::QUIT(message.to_string()))
                .await?;
        } else {
            self.write_command(commands::Command::QUIT(format!(
                "circe {} (https://crates.io/crates/circe)",
                env!("CARGO_PKG_VERSION")
            )))
            .await?;
        }
        tracing::info!("circe shutting down");
        self.stream.shutdown().await?;
        Ok(())
    }
}

impl Config {
    /// Create a new config for the client
    ///
    /// ```rust
    /// # use circe::*;
    /// let config = Config::new(
    ///     &["#main", "#circe"],
    ///     "karx.xyz",
    ///     Some("+B"),
    ///     Some("circe"),
    ///     6697,
    ///     "circe",
    /// );
    /// ```
    pub fn new(
        channels: &[&'static str],
        host: &str,
        mode: Option<&'static str>,
        nickname: Option<&'static str>,
        port: u16,
        username: &str,
    ) -> Self {
        tracing::info!("New circe client config");
        // Conversion from &'static str to String
        let channels_config = channels
            .iter()
            .map(|channel| (*channel).to_string())
            .collect();

        let mode_config: Option<String>;
        if let Some(mode) = mode {
            mode_config = Some(mode.to_string());
        } else {
            mode_config = None;
        }

        let nickname_config: Option<String>;
        if let Some(nickname) = nickname {
            nickname_config = Some(nickname.to_string());
        } else {
            nickname_config = Some(username.to_string());
        }

        let config = Self {
            channels: channels_config,
            host: host.into(),
            mode: mode_config,
            nickname: nickname_config,
            port,
            username: username.into(),
        };

        tracing::debug!("circe config: {:?}", config);
        config
    }

    /// Allows for configuring circe at runtime
    pub fn runtime_config(
        channels: Vec<String>,
        host: String,
        mode: Option<String>,
        nickname: Option<String>,
        port: u16,
        username: String,
    ) -> Self {
        tracing::info!("New circe client config");
        let mode_config: Option<String>;
        if let Some(mode) = mode {
            mode_config = Some(mode.to_string());
        } else {
            mode_config = None;
        }

        let nickname_config: Option<String>;
        if let Some(nickname) = nickname {
            nickname_config = Some(nickname.to_string());
        } else {
            nickname_config = Some(username.to_string());
        }

        let config = Self {
            channels,
            host,
            mode: mode_config,
            nickname: nickname_config,
            port,
            username,
        };

        tracing::debug!("circe config: {:?}", config);
        config
    }

    /// Create a config from a toml file
    /// ```toml
    /// channels = ["#main", "#main2"]
    /// host = "192.168.178.100"
    /// mode = "+B"
    /// nickname = "circe"
    /// port = 6667
    /// username = "circe"
    /// ```
    /// # Errors
    /// Returns an Error if the file cannot be opened or if the TOML is invalid
    ///
    #[cfg_attr(docsrs, doc(cfg(feature = "toml_support")))]
    #[cfg(feature = "toml_support")]
    pub fn from_toml<P: AsRef<Path>>(path: P) -> Result<Self, Error> {
        tracing::info!("New circe client config");
        use std::io::Read;
        let mut file = File::open(&path)?;
        let mut data = String::new();
        file.read_to_string(&mut data)?;

        let config = toml::from_str(&data).map_err(|e| {
            use tokio::io::ErrorKind;
            Error::new(ErrorKind::Other, format!("Invalid TOML: {}", e))
        });
        tracing::debug!("circe config: {:?}", config);
        config
    }
}
