use stubr::Stubr;
use surf::get;

use asserhttp::*;

mod eq {
    use super::*;

    #[async_std::test]
    async fn should_expect_header() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header("x-a", "a");
    }

    #[async_std::test]
    async fn should_expect_header_ignoring_case() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header("X-A", "a");
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'x-a' to be equal to 'A' but was 'a'")]
    async fn expect_header_value_should_be_case_sensitive() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header("x-a", "A");
    }

    #[async_std::test]
    async fn should_expect_many_header() {
        let srv = Stubr::start("tests/stubs/header/many.json").await;
        get(&srv.uri()).await.unwrap()
            .expect_header("x-a", "a")
            .expect_header("x-b", "b");
    }

    #[async_std::test]
    #[should_panic(expected = "expected one header named 'x-b' but none found")]
    async fn expect_header_should_panic_when_wrong_key() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header("x-b", "a");
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'x-a' to be equal to 'b' but was 'a'")]
    async fn expect_header_should_panic_when_wrong_value() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header("x-a", "b");
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'x-m' to be single valued. Had '2' values '[\"a\", \"b\"]'. Use 'expect_headers' instead.")]
    async fn expect_header_should_panic_when_multi_valued() {
        let srv = Stubr::start("tests/stubs/header/multi.json").await;
        get(&srv.uri()).await.unwrap().expect_header("x-m", "a");
    }

    #[async_std::test]
    async fn result_should_expect_header() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header("x-a", "a");
    }

    #[async_std::test]
    async fn result_should_expect_many_header() {
        let srv = Stubr::start("tests/stubs/header/many.json").await;
        get(&srv.uri()).await
            .expect_header("x-a", "a")
            .expect_header("x-b", "b");
    }

    #[async_std::test]
    #[should_panic(expected = "expected one header named 'x-b' but none found")]
    async fn result_expect_header_should_panic_when_wrong_key() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header("x-b", "a");
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'x-a' to be equal to 'b' but was 'a'")]
    async fn result_expect_header_should_panic_when_wrong_value() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header("x-a", "b");
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'x-m' to be single valued. Had '2' values '[\"a\", \"b\"]'. Use 'expect_headers' instead.")]
    async fn result_expect_header_should_panic_when_multi_valued() {
        let srv = Stubr::start("tests/stubs/header/multi.json").await;
        get(&srv.uri()).await.expect_header("x-m", "a");
    }
}

mod present {
    use super::*;

    #[async_std::test]
    async fn should_expect_header_present() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header_present("x-a");
    }

    #[async_std::test]
    async fn should_expect_header_present_ignoring_case() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header_present("X-A");
    }

    #[async_std::test]
    #[should_panic(expected = "expected one header named 'x-b' but none found")]
    async fn expect_header_present_should_fail_when_absent() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header_present("x-b");
    }

    #[async_std::test]
    async fn result_should_expect_header_present() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header_present("x-a");
    }

    #[async_std::test]
    async fn result_should_expect_header_present_ignoring_case() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header_present("X-A");
    }

    #[async_std::test]
    #[should_panic(expected = "expected one header named 'x-b' but none found")]
    async fn result_expect_header_present_should_fail_when_absent() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header_present("x-b");
    }
}

mod absent {
    use super::*;

    #[async_std::test]
    async fn should_expect_header_absent() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header_absent("x-b");
    }

    #[async_std::test]
    #[should_panic(expected = "expected no header named 'x-a' but one found")]
    async fn expect_header_absent_should_fail_when_absent() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header_absent("x-a");
    }

    #[async_std::test]
    #[should_panic(expected = "expected no header named 'x-a' but one found")]
    async fn expect_header_absent_should_ignore_case() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.unwrap().expect_header_absent("X-A");
    }

    #[async_std::test]
    async fn result_should_expect_header_absent() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header_absent("x-b");
    }

    #[async_std::test]
    #[should_panic(expected = "expected no header named 'x-a' but one found")]
    async fn result_expect_header_absent_should_fail_when_absent() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header_absent("x-a");
    }

    #[async_std::test]
    #[should_panic(expected = "expected no header named 'x-a' but one found")]
    async fn result_expect_header_absent_should_ignore_case() {
        let srv = Stubr::start("tests/stubs/header/one.json").await;
        get(&srv.uri()).await.expect_header_absent("X-A");
    }
}

mod json {
    use super::*;

    #[async_std::test]
    async fn should_expect_content_type_json() {
        let srv = Stubr::start("tests/stubs/header/json.json").await;
        get(&srv.uri()).await.unwrap().expect_content_type_json();
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'content-type' to be equal to 'application/json' but was 'application/xml'")]
    async fn expect_header_should_panic_when_wrong_value() {
        let srv = Stubr::start("tests/stubs/header/xml.json").await;
        get(&srv.uri()).await.unwrap().expect_content_type_json();
    }

    #[async_std::test]
    async fn result_should_expect_content_type_json() {
        let srv = Stubr::start("tests/stubs/header/json.json").await;
        get(&srv.uri()).await.expect_content_type_json();
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'content-type' to be equal to 'application/json' but was 'application/xml'")]
    async fn result_expect_header_should_panic_when_wrong_value() {
        let srv = Stubr::start("tests/stubs/header/xml.json").await;
        get(&srv.uri()).await.expect_content_type_json();
    }
}

mod text {
    use super::*;

    #[async_std::test]
    async fn should_expect_content_type_text() {
        let srv = Stubr::start("tests/stubs/header/text.json").await;
        get(&srv.uri()).await.unwrap().expect_content_type_text();
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'content-type' to be equal to 'text/plain' but was 'application/xml'")]
    async fn expect_header_should_panic_when_wrong_value() {
        let srv = Stubr::start("tests/stubs/header/xml.json").await;
        get(&srv.uri()).await.unwrap().expect_content_type_text();
    }

    #[async_std::test]
    async fn result_should_expect_content_type_text() {
        let srv = Stubr::start("tests/stubs/header/text.json").await;
        get(&srv.uri()).await.expect_content_type_text();
    }

    #[async_std::test]
    #[should_panic(expected = "expected header 'content-type' to be equal to 'text/plain' but was 'application/xml'")]
    async fn result_expect_header_should_panic_when_wrong_value() {
        let srv = Stubr::start("tests/stubs/header/xml.json").await;
        get(&srv.uri()).await.expect_content_type_text();
    }
}