pub mod root {
    use std::rc::Rc;

    use clap::crate_version;

    use crate::providers::*;
    use crate::transpiler::{asml, hcl, Artifact, ArtifactError, ContentType};

    pub type ArtifactList = Vec<Box<hcl::service::Module>>;

    pub struct Module {
        hcl: Rc<Option<String>>,
        ctx: Rc<asml::Context>,
        pub services: Option<ArtifactList>,
    }

    impl Module {
        pub fn new(ctx: Rc<asml::Context>) -> Self {
            Self { hcl: Rc::new(None), services: None, ctx }
        }
    }

    impl Artifact for Module {
        fn content_type(&self) -> ContentType {
            ContentType::HCL("HCL")
        }

        fn content(&self) -> Rc<Option<String>> {
            self.hcl.clone()
        }

        fn cast(&mut self) -> Result<String, ArtifactError> {
            let mut content = format!("# Generated by assemblylift-cli {}\r\n", crate_version!());

            let provider = (*ROOT_PROVIDERS.get("root").expect("could not find root provider")).lock().unwrap();
            let artifact = provider.transform(self.ctx.clone(), String::from("root"))
                .expect("unexpected error transforming root manifest");
            let artifact_content = artifact.content().clone();
            content.push_str(&artifact_content.as_ref().as_ref().unwrap());

            let services = &self.ctx.clone().services;
            let mut service_modules: ArtifactList = ArtifactList::new();
            for service in services {
                let mut module = hcl::service::Module::new(self.ctx.clone(), service.name.clone());
                content.push_str(&module.cast().unwrap());
                service_modules.push(Box::new(module));
            }

            self.services = Some(service_modules);
            self.hcl = Rc::new(Some(content.clone()));
            Ok(content)
        }
    }
}

pub mod service {
    use std::rc::Rc;

    use crate::providers::*;
    use crate::transpiler::{asml, hcl, toml, Artifact, ArtifactError, ContentType, StringMap};

    pub type ArtifactList = Vec<Box<hcl::function::Module>>;

    pub struct Module {
        hcl: Rc<Option<String>>,
        ctx: Rc<asml::Context>,
        name: String,
        pub functions: Option<ArtifactList>,
        pub iomods: Option<StringMap<toml::service::Dependency>>,
    }

    impl Module {
        pub fn new(ctx: Rc<asml::Context>, name: String) -> Self {
            Module { hcl: Rc::new(None), ctx, name, functions: None, iomods: None }
        }
    }

    impl Artifact for Module {
        fn content_type(&self) -> ContentType {
            ContentType::HCL("HCL")
        }

        fn content(&self) -> Rc<Option<String>> {
            self.hcl.clone()
        }

        fn cast(&mut self) -> Result<String, ArtifactError> {
            let service_name = self.name.clone();
            let mut content = format!("# Begin service `{}`\n", service_name.clone());

            match self.ctx.services.iter().find(|&s| *s.name == self.name.clone()) {
                Some(service) => {
                    let functions = &self.ctx.functions;

                    let provider_name = service.provider.name.clone();
                    let mut service_provider = SERVICE_PROVIDERS.get(&provider_name)
                        .expect(&format!("could not find service provider named {}", provider_name))
                        .lock()
                        .unwrap();

                    service_provider.set_options(service.provider.options.clone())
                        .expect("unable to set provider options");
                    service_provider.init(self.ctx.clone(), service_name.clone())
                        .expect("unable to initialize service provider");

                    let service_artifact = service_provider.transform(self.ctx.clone(), service_name.clone())
                        .expect("unexpected error transforming service");
                    content.push_str(&service_artifact.content().as_ref().as_ref().unwrap());

                    let mut function_modules: ArtifactList = ArtifactList::new();
                    for function in functions.iter().filter(|&f| *f.service_name == service_name.clone()) {
                        let mut module = hcl::function::Module::new(self.ctx.clone(), function.name.clone());
                        content.push_str(&module.cast().unwrap());
                        function_modules.push(Box::new(module));
                    }

                    self.hcl = Rc::new(Some(content.clone()));
                    self.functions = Some(function_modules);
                    Ok(content)
                }
                None => Err(ArtifactError(format!("unable to find service {} in context", self.name.clone()))),
            }
        }
    }
}

pub mod function {
    use std::rc::Rc;

    use crate::providers::*;
    use crate::transpiler::{asml, Artifact, ArtifactError, ContentType};

    pub struct Module {
        hcl: Rc<Option<String>>,
        ctx: Rc<asml::Context>,
        name: String,
    }

    impl Module {
        pub fn new(ctx: Rc<asml::Context>, name: String) -> Self {
            Self { hcl: Rc::new(None), ctx, name }
        }
    }

    impl Artifact for Module {
        fn content_type(&self) -> ContentType {
            ContentType::HCL("HCL")
        }

        fn content(&self) -> Rc<Option<String>> {
            self.hcl.clone()
        }

        fn cast(&mut self) -> Result<String, ArtifactError> {
            match self.ctx.functions.iter().find(|&f| *f.name == self.name.clone()) {
                Some(function) => {
                    let provider_name = function.provider.name.clone();
                    let mut function_provider = FUNCTION_PROVIDERS.get(&provider_name)
                        .expect(&format!("could not find function provider named {}", provider_name))
                        .lock()
                        .unwrap();

                    function_provider.set_options(function.provider.options.clone())
                        .expect("unable to set function provider options");
                    function_provider.init(self.ctx.clone(), function.name.clone())
                        .expect("unable to initialize function provider");

                    let mut function_artifact = function_provider.transform(self.ctx.clone(), function.name.clone())
                        .expect("unexpected error transforming function");
                    let content = function_artifact.cast().unwrap();
                    self.hcl = Rc::new(Some(content.clone()));
                    Ok(content)
                }
                None => Err(ArtifactError(format!("unable to find function {} in context", self.name.clone()))),
            }
        }
    }
}
