use wither::mongodb::Database;

pub type JobId = String;

#[cfg(feature = "bill")]
pub mod bill;
pub mod plaid_transaction;
#[cfg(feature = "user")]
pub mod user;

// wither::mongodb::bson::ObjectId's cant be imported in target wasm32_unknown_unknown
// becacuse wither depends on socket2, which relies on code not available on that target
#[cfg(target_arch = "wasm32")]
pub type Id = String;
#[cfg(not(target_arch = "wasm32"))]
pub type Id = wither::mongodb::bson::oid::ObjectId;

cfg_if::cfg_if! {
    if #[cfg(not(target_arch="wasm32"))]
    {
        // use wither::bson::{doc, oid::ObjectId};
        use wither::mongodb::Client;
        // use wither::prelude::*;
        use wither::Model;

    use wither::mongodb::options::ClientOptions;
        pub const ENV_PREFIX_MONGODB: &'static str = "MONGODB_";
        pub const ENV_KEY_URL: &'static str = "URL";
        pub const ENV_KEY_DATABASE_NAME:&'static str = "DATABASE_NAME";
        pub const ENV_KEY_MONGODB_MAX_CONNECTIONS: &'static str = "MAX_CONNECTIONS";
        pub const DEFAULT_DATABASE_NAME:&'static str = "db";
        pub const ENV_DEFAULT_MONGODB_MAX_CONNECTIONS: u32 = 5;
    }

}

// mongo clients cant be created in webassembly
#[cfg(not(target_arch = "wasm32"))]
pub async fn create_mongo_client() -> Result<(Client, ClientOptions), crate::Error> {
    let connection_uri = std::env::var(&format!("{}{}", ENV_PREFIX_MONGODB, ENV_KEY_URL))?;
    let max_connections_string = std::env::var(&format!(
        "{}{}",
        ENV_PREFIX_MONGODB, ENV_KEY_MONGODB_MAX_CONNECTIONS
    ))
    .unwrap_or(ENV_DEFAULT_MONGODB_MAX_CONNECTIONS.to_string());
    let max_connections: u32 = max_connections_string
        .as_str()
        .parse::<u32>()
        .unwrap_or(ENV_DEFAULT_MONGODB_MAX_CONNECTIONS);

    // let MAX_CONNECTIONS
    // let client = Client::with_uri_str(&connection_uri).await?;
    let mut options: wither::mongodb::options::ClientOptions =
        wither::mongodb::options::ClientOptions::parse(&connection_uri).await?;
    if (options.max_pool_size).is_none() {
        options.max_pool_size = Some(max_connections);
    };
    // let database= client.database("");

    Ok((Client::with_options(options.clone())?, options))
}
// it is impossible to perform mongodb related operations directly in wasm
#[cfg(not(target_arch = "wasm32"))]
pub async fn sync_indexes(db: &wither::mongodb::Database) -> Result<(), crate::Error> {
    #[cfg(feature = "log")]
    log::debug!("Syncing Indexes...");
    #[cfg(feature = "user")]
    user::User::sync(db).await?;
    // plaid_transaction::PlaidTransaction::sync(db).await?;
    Ok(())
}

pub fn get_system_datbase_name_from_env() -> String {
    let key = format!("{}{}", ENV_PREFIX_MONGODB, ENV_KEY_DATABASE_NAME);
    return match std::env::var(key) {
        Ok(var) => var,
        Err(std::env::VarError::NotPresent) => {
            #[cfg(feature = "enable-log")]
            {
                log::info!(
                    "ENV key '{}' not present. using default database '{}'",
                    key,
                    DEFAULT_DATABASE_NAME
                );
            };
            DEFAULT_DATABASE_NAME.to_string()
        }
        Err(std::env::VarError::NotUnicode(_)) => {
            #[cfg(feature = "enable-log")]
            {
                log::info!("ENV key '{}' was present, but did not contain valid unicode. using default database '{}'",key, DEFAULT_DATABASE_NAME);
            };
            DEFAULT_DATABASE_NAME.to_string()
        }
    };
}
pub fn get_system_database(client: &wither::mongodb::Client) -> Database {
    client.database(&get_system_datbase_name_from_env())
}

#[cfg(test)]
#[cfg(not(target_arch = "wasm32"))]
pub mod mongo_tests {

    use super::{create_mongo_client, get_system_datbase_name_from_env, sync_indexes};
    #[tokio::test]
    pub async fn test_create_mongo_client() -> Result<(), crate::Error> {
        let (client, _options) = create_mongo_client().await?;
        // options.driver_info.
        let database = client.database(&get_system_datbase_name_from_env());
        sync_indexes(&database).await?;
        Ok(())
    }
}
