use super::plaid_transaction::PlaidTransaction;
use serde::{Deserialize, Serialize};

pub type Id = String;
cfg_if::cfg_if! {
    if #[cfg(target_arch="wasm32")] {
        pub type Oid = String;
    }
    else {
        use wither::prelude::*;
        // use wither::Model;
        use wither::bson::doc;
        pub type Oid = wither::mongodb::bson::oid::ObjectId;
    }
}

#[derive(Serialize, Deserialize)]
#[cfg_attr(not(target_arch = "wasm32"), derive(wither::Model))]
#[cfg_attr(
    not(target_arch = "wasm32"),
    model(
        index(keys = r#"doc!{"idp_id": 1}"#, options = r#"doc!{"unique": true}"#),
        read_concern = "majority"
    )
)]
#[cfg_attr(target_arch = "wasm32", wasm_bindgen)]
pub struct User {
    access_token: Option<String>,
    #[serde(rename = "_id", skip_serializing_if = "Option::is_none")]
    id: Option<Oid>,
    idp_id: Id,
    // plaid_transactions: Vec<PlaidTransaction>,
}
impl User {
    pub fn access_token(&self) -> Option<String> {
        self.access_token.to_owned()
    }
    pub fn id(&self) -> Option<Oid> {
        self.id.to_owned()
    }
    pub fn idp_id(&self) -> &str {
        self.idp_id.as_str()
    }
    // pub fn plaid_transactions(&self) -> &Vec<PlaidTransaction> {
    //     &self.plaid_transactions
    // }
}
#[cfg(all(not(target_arch="wasm32"),feature = "napi"))]
impl User {
    pub fn to_object<'a>(
        &self,
        cx: &mut impl neon::context::Context<'a>,
    ) -> neon::result::JsResult<'a, neon::types::JsObject> {
        use neon::object::Object;
        let user = cx.empty_object();
        match self.access_token {
            Some(ref at) => {
                    let access_token = cx.string(at);
                    user.set(cx, "access_token", access_token)?;
                
            }
            None => {
                let access_token = cx.null();
                user.set(cx, "access_token", access_token)?;
            }
        }
        match self.id {
            Some(ref id) => {
                let id = cx.string(id.to_string());
                user.set(cx, "_id", id)?;
            }
            None => {
                let id = cx.null();
                user.set(cx, "_id", id)?;
            }
        }
        let idp_id = cx.string(self.idp_id.to_owned());

        user.set(cx, "idp_id", idp_id)?;
        Ok(user)
    }
}
cfg_if::cfg_if!{
    if #[cfg(feature="gql")] {
        // create a Graphql Wrapper object that holds a reference to a user object
        pub struct GQLUser<'a>(pub &'a User);
        #[async_graphql::Object]
        impl<'a> GQLUser<'a> {
            pub async fn id(&self)-> Option<String> {
                match self.0.id {
                    Some(id)=>Some(id.to_string()),
                    None=>None
                }
            }
        }
    }
}