use std::collections::HashMap;

use asml_iomod_dynamodb::query;
use asml_iomod_dynamodb::structs::*;

use crate::error::DynamoDbError;

pub async fn get_module(
    artifact_coords: String, 
    artifact_version: String, 
) -> Result<GetModuleResponse, DynamoDbError> {
    let mut input: QueryInput = Default::default();
    input.table_name = String::from("asml-iomod-registry");
    input.index_name = Some(String::from("sk-index"));
    input.key_condition_expression = Some(String::from("#version = :version AND #fullcoords = :fullcoords"));

    let mut attribute_names: HashMap<String, AttributeName> = HashMap::new();
    attribute_names.insert(String::from("#version"), String::from("sk"));
    attribute_names.insert(String::from("#fullcoords"), String::from("data"));
    input.expression_attribute_names = Some(attribute_names);

    let mut attribute_values: HashMap<String, AttributeValue> = HashMap::new();
    let mut version_value: AttributeValue = Default::default();
    version_value.s = Some("MODULE:version".to_string());
    attribute_values.insert(String::from(":version"), version_value);
    let mut fullcoords_value: AttributeValue = Default::default();
    fullcoords_value.s = Some(format!("{}#{}", 
            artifact_coords.clone(), 
            artifact_version.clone(),
    ));
    attribute_values.insert(String::from(":fullcoords"), fullcoords_value);
    input.expression_attribute_values = Some(attribute_values);

    match query(input).await {
        Ok(response) => {
            let items = response.items.unwrap();
            if items.len() > 0 {
                Ok(GetModuleResponse {
                    uuid: String::from(items[0].get("pk").unwrap().s.as_ref().unwrap().split('#').collect::<Vec<&str>>()[1]),
                    user_id: String::from(items[0].get("user").unwrap().s.as_ref().unwrap()),
                    coordinates: String::from(items[0].get("data").unwrap().s.as_ref().unwrap().split('#').collect::<Vec<&str>>()[0]),
                    version: String::from(items[0].get("data").unwrap().s.as_ref().unwrap().split('#').collect::<Vec<&str>>()[1]),
                })
            } else {
                Err(DynamoDbError { why: format!("no module found at {}@{}", artifact_coords.clone(), artifact_version.clone()) })
            }
        },
        Err(err) => Err(DynamoDbError { why: err.why }), 
    }
}

#[derive(Debug)]
pub struct GetModuleResponse {
    pub uuid: String,
    pub user_id: String,
    pub coordinates: String,
    pub version: String,
}
