use super::{Csr, Fence, FenceType, Instruction, Register};

impl std::fmt::Display for Register {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Register::X0 => f.write_str("zero"),
            Register::X1 => f.write_str("ra"),
            Register::X2 => f.write_str("sp"),
            Register::X3 => f.write_str("gp"),
            Register::X4 => f.write_str("tp"),
            Register::X5 => f.write_str("t0"),
            Register::X6 => f.write_str("t1"),
            Register::X7 => f.write_str("t2"),
            Register::X8 => f.write_str("s0"),
            Register::X9 => f.write_str("s1"),
            Register::X10 => f.write_str("a0"),
            Register::X11 => f.write_str("a1"),
            Register::X12 => f.write_str("a2"),
            Register::X13 => f.write_str("a3"),
            Register::X14 => f.write_str("a4"),
            Register::X15 => f.write_str("a5"),
            Register::X16 => f.write_str("a6"),
            Register::X17 => f.write_str("a7"),
            Register::X18 => f.write_str("s2"),
            Register::X19 => f.write_str("s3"),
            Register::X20 => f.write_str("s4"),
            Register::X21 => f.write_str("s5"),
            Register::X22 => f.write_str("s6"),
            Register::X23 => f.write_str("s7"),
            Register::X24 => f.write_str("s8"),
            Register::X25 => f.write_str("s9"),
            Register::X26 => f.write_str("s10"),
            Register::X27 => f.write_str("s11"),
            Register::X28 => f.write_str("t3"),
            Register::X29 => f.write_str("t4"),
            Register::X30 => f.write_str("t5"),
            Register::X31 => f.write_str("t6"),
        }
    }
}

impl std::fmt::Display for Csr {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Csr::FFlags => todo!(),
            Csr::FRm => todo!(),
            Csr::Fcsr => todo!(),
            Csr::Cycle => write!(f, "cycle"),
            Csr::Time => todo!(),
            Csr::InstRet => todo!(),
            Csr::HPMCounter(_) => todo!(),
            Csr::CycleH => todo!(),
            Csr::TimeH => todo!(),
            Csr::InstRetH => todo!(),
            Csr::HPMCounterH(_) => todo!(),
            Csr::SStatus => todo!(),
            Csr::Sie => todo!(),
            Csr::STVec => todo!(),
            Csr::SCounterEn => todo!(),
            Csr::SEnvCfg => todo!(),
            Csr::SScratch => todo!(),
            Csr::Sepc => todo!(),
            Csr::Scause => todo!(),
            Csr::Stval => todo!(),
            Csr::Sip => todo!(),
            Csr::Satp => todo!(),
            Csr::SContext => todo!(),
            Csr::Other(_) => todo!(),
        }
    }
}

impl std::fmt::Display for FenceType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        if self.contains(FenceType::WRITES) {
            f.write_str("w")?;
        }

        if self.contains(FenceType::READS) {
            f.write_str("r")?;
        }

        if self.contains(FenceType::OUTPUT) {
            f.write_str("o")?;
        }

        if self.contains(FenceType::INPUT) {
            f.write_str("i")?;
        }

        Ok(())
    }
}

impl std::fmt::Display for Fence {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match *self {
            Fence::General { succ, pred } => write!(f, "fence {}, {}", succ, pred)?,
            Fence::TSO => todo!(),
            Fence::PAUSE => write!(f, "pause")?,
            Fence::I(_, _, _) => todo!(),
            Fence::Unknown {
                rd,
                rs1,
                funct3,
                succ,
                pred,
                fm,
            } => todo!(),
        }
        Ok(())
    }
}

impl std::fmt::Display for Instruction {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match *self {
            Instruction::Lui(rd, imm) => write!(f, "lui {}, {}", rd, imm >> 12)?,
            Instruction::AuiPC(rd, imm) => write!(f, "auipc {}, {}", rd, imm >> 12)?,
            Instruction::Jal(rd, imm) => {
                if rd == Register::X0 {
                    write!(f, "j pc + {}", imm)?;
                } else {
                    write!(f, "jal {}, pc + {}", rd, imm)?;
                }
            }
            Instruction::Jalr(rd, rs1, imm) => {
                if rd == Register::X0 && rs1 == Register::X1 && imm == 0 {
                    write!(f, "ret")?;
                } else if rd == Register::X0 {
                    write!(f, "jr {}({})", imm, rs1)?;
                } else {
                    write!(f, "jalr {}, {}({})", rd, imm, rs1)?;
                }
            }
            Instruction::Beq(rs1, rs2, imm) => write!(f, "beq {}, {}, {}(pc)", rs1, rs2, imm)?,
            Instruction::Bne(rs1, rs2, imm) => write!(f, "bne {}, {}, {}(pc)", rs1, rs2, imm)?,
            Instruction::Blt(rs1, rs2, imm) => write!(f, "blt {rs1}, {rs2}, {imm}(pc)")?,
            Instruction::Bge(rs1, rs2, imm) => write!(f, "bge {rs1}, {rs2}, {imm}(pc)")?,
            Instruction::Bltu(rs1, rs2, imm) => write!(f, "bltu {}, {}, {}(pc)", rs1, rs2, imm)?,
            Instruction::Bgeu(rs1, rs2, imm) => write!(f, "bgeu {}, {}, {}(pc)", rs1, rs2, imm)?,
            Instruction::Lb(rd, rs1, imm) => write!(f, "lb {}, {}({})", rd, imm, rs1)?,
            Instruction::Lh(_, _, _) => todo!(),
            Instruction::Lw(rd, rs1, imm) => write!(f, "lw {}, {}({})", rd, imm, rs1)?,
            Instruction::Ld(rd, rs1, imm) => write!(f, "ld {}, {}({})", rd, imm, rs1)?,
            Instruction::Lbu(rd, rs1, imm) => write!(f, "lbu {}, {}({})", rd, imm, rs1)?,
            Instruction::Lhu(_, _, _) => todo!(),
            Instruction::Lwu(rd, rs1, imm) => write!(f, "lwu {}, {}({})", rd, imm, rs1)?,
            Instruction::Sb(rd, rs1, imm) => write!(f, "sb {}, {}({})", rd, imm, rs1)?,
            Instruction::Sh(_, _, _) => todo!(),
            Instruction::Sw(rd, rs1, imm) => write!(f, "sw {}, {}({})", rd, imm, rs1)?,
            Instruction::Sd(rd, rs1, imm) => write!(f, "sd {}, {}({})", rd, imm, rs1)?,
            Instruction::Addi(rd, rs1, imm) => {
                if imm == 0 {
                    write!(f, "mv {}, {}", rd, rs1)?;
                } else {
                    write!(f, "addi {}, {}, {}", rd, rs1, imm)?;
                }
            }
            Instruction::Addiw(rd, rs1, imm) => {
                if imm == 0 {
                    write!(f, "sext.w {}, {}", rd, rs1)?;
                } else {
                    write!(f, "addi.w {}, {}, {}", rd, rs1, imm)?;
                }
            }
            Instruction::Slti(_, _, _) => todo!(),
            Instruction::Sltiu(rd, rs1, imm) => write!(f, "sltiu {rd}, {rs1}, {imm}")?,
            Instruction::Xori(rd, rs1, imm) => {
                if imm == -1 {
                    write!(f, "not {rd}, {rs1}")?;
                } else {
                    write!(f, "xori {rd}, {rs1}, {imm}")?;
                }
            }
            Instruction::ORI(rd, rs1, imm) => write!(f, "ori {rd}, {rs1}, {imm}")?,
            Instruction::ANDI(rd, rs1, imm) => write!(f, "andi {rd}, {rs1}, {imm}")?,
            Instruction::SLLI(rd, rs1, imm) => write!(f, "slli {rd}, {rs1}, {imm}")?,
            Instruction::SLLIW(rd, rs1, imm) => write!(f, "slliw {rd}, {rs1}, {imm}")?,
            Instruction::SRLI(rd, rs1, imm) => write!(f, "srli {rd}, {rs1}, {imm}")?,
            Instruction::SRLIW(rd, rs1, imm) => write!(f, "srliw {rd}, {rs1}, {imm}")?,
            Instruction::SRAI(_, _, _) => todo!(),
            Instruction::SRAIW(_, _, _) => todo!(),
            Instruction::ADD(rd, rs1, rs2) => write!(f, "add {}, {}, {}", rd, rs1, rs2)?,
            Instruction::ADDW(rd, rs1, rs2) => write!(f, "addw {}, {}, {}", rd, rs1, rs2)?,
            Instruction::SUB(rd, rs1, rs2) => {
                if rs1 == Register::X0 {
                    write!(f, "neg {rd}, {rs2}")?;
                } else if rs2 == Register::X0 {
                    write!(f, "neg {rd}, {rs1}")?;
                } else {
                    write!(f, "sub {rd}, {rs1}, {rs2}")?;
                }
            }
            Instruction::SUBW(rd, rs1, rs2) => write!(f, "subw {rd}, {rs1}, {rs2}")?,
            Instruction::SLL(rd, rs1, rs2) => write!(f, "sll {rd}, {rs1}, {rs2}")?,
            Instruction::SLLW(_, _, _) => todo!(),
            Instruction::SLT(_, _, _) => todo!(),
            Instruction::SLTU(rd, rs1, rs2) => write!(f, "sltu {rd}, {rs1}, {rs2}")?,
            Instruction::XOR(rd, rs1, rs2) => write!(f, "xor {rd}, {rs1}, {rs2}")?,
            Instruction::SRL(rd, rs1, rs2) => write!(f, "srl {rd}, {rs1}, {rs2}")?,
            Instruction::SRLW(_, _, _) => todo!(),
            Instruction::SRA(_, _, _) => todo!(),
            Instruction::SRAW(_, _, _) => todo!(),
            Instruction::OR(rd, rs1, rs2) => write!(f, "or {rd}, {rs1}, {rs2}")?,
            Instruction::AND(rd, rs1, rs2) => write!(f, "and {rd}, {rs1}, {rs2}")?,
            Instruction::Mul(rd, rs1, rs2) => write!(f, "mul {rd}, {rs1}, {rs2}")?,
            Instruction::Mulh(_, _, _) => todo!(),
            Instruction::Mulhsu(_, _, _) => todo!(),
            Instruction::Mulhu(_, _, _) => todo!(),
            Instruction::Mulw(_, _, _) => todo!(),
            Instruction::Div(_, _, _) => todo!(),
            Instruction::Divu(_, _, _) => todo!(),
            Instruction::Divw(_, _, _) => todo!(),
            Instruction::Divuw(_, _, _) => todo!(),
            Instruction::Rem(_, _, _) => todo!(),
            Instruction::Remu(_, _, _) => todo!(),
            Instruction::Remw(_, _, _) => todo!(),
            Instruction::Remuw(_, _, _) => todo!(),
            Instruction::FENCE(ref fence) => fence.fmt(f)?,
            Instruction::ECALL => write!(f, "ecall")?,
            Instruction::EBREAK => todo!(),
            Instruction::CsrRw(rd, rs1, reg) => write!(f, "csrrw {rd}, {rs1}, {reg}")?,
            Instruction::CsrRs(_, _, _) => todo!(),
            Instruction::CsrRc(_, _, _) => todo!(),
            Instruction::CsrRwi(_, _, _) => todo!(),
            Instruction::CsrRsi(_, _, _) => todo!(),
            Instruction::CsrRci(_, _, _) => todo!(),
            Instruction::LrW(_, _, _, _) => todo!(),
            Instruction::ScW(_, _, _, _) => todo!(),
            Instruction::AMOSwapW(_, _, _, _) => todo!(),
            Instruction::AMOAddW(_, _, _, _) => todo!(),
            Instruction::AMOXorW(_, _, _, _) => todo!(),
            Instruction::AMOAndW(_, _, _, _) => todo!(),
            Instruction::AMOOrW(_, _, _, _) => todo!(),
            Instruction::AMOMinW(_, _, _, _) => todo!(),
            Instruction::AMOMaxW(_, _, _, _) => todo!(),
            Instruction::AMOMinUW(_, _, _, _) => todo!(),
            Instruction::AMOMaxUW(_, _, _, _) => todo!(),
            Instruction::LrD(_, _, _, _) => todo!(),
            Instruction::ScD(_, _, _, _) => todo!(),
            Instruction::AMOSwapD(_, _, _, _) => todo!(),
            Instruction::AMOAddD(_, _, _, _) => todo!(),
            Instruction::AMOXorD(_, _, _, _) => todo!(),
            Instruction::AMOAndD(_, _, _, _) => todo!(),
            Instruction::AMOOrD(_, _, _, _) => todo!(),
            Instruction::AMOMinD(_, _, _, _) => todo!(),
            Instruction::AMOMaxD(_, _, _, _) => todo!(),
            Instruction::AMOMinUD(_, _, _, _) => todo!(),
            Instruction::AMOMaxUD(_, _, _, _) => todo!(),
        }

        Ok(())
    }
}
