mod decode;
//mod print;

pub use decode::Decoder;

use std::sync::atomic::Ordering;

pub enum FloatFormat {
    Single,
    Double,
    Half,
    Quad,
}

impl FloatFormat {
    fn from_inst(inst: u32) -> Self {
        let funct2 = (inst >> 25) & 0b11;

        match funct2 {
            0b00 => Self::Single,
            0b01 => Self::Double,
            0b10 => Self::Half,
            0b11 => Self::Quad,
            _ => unreachable!(),
        }
    }
}

#[derive(Debug, Clone, Copy)]
pub enum Csr {
    // USER Mode start
    FFlags,
    FRm,
    Fcsr,

    Cycle,
    Time,
    InstRet,
    HPMCounter(u8),

    CycleH,
    TimeH,
    InstRetH,
    HPMCounterH(u8),
    // USER Mode end

    // SUPERVISOR Mode start
    SStatus,
    Sie,
    STVec,
    SCounterEn,

    SEnvCfg,

    SScratch,
    Sepc,
    Scause,
    Stval,
    Sip,

    Satp,

    SContext,
    // SUPERVISOR Mode end

    // Other
    Other(u16),
}

impl From<u16> for Csr {
    fn from(csr: u16) -> Self {
        match csr {
            // USER Mode start
            0x001 => Self::FFlags,
            0x002 => Self::FRm,
            0x003 => Self::Fcsr,

            0xC00 => Self::Cycle,
            0xC01 => Self::Time,
            0xC02 => Self::InstRet,
            0xC03..=0xC1F => Self::HPMCounter((csr & 0xFF) as u8),

            0xC80 => Self::CycleH,
            0xC81 => Self::TimeH,
            0xC82 => Self::InstRetH,
            0xC83..=0xC9F => Self::HPMCounterH((csr & 0xFF) as u8),
            // USER Mode end

            // SUPERVISOR Mode start
            0x100 => Self::SStatus,
            0x104 => Self::Sie,
            0x105 => Self::STVec,
            0x106 => Self::SCounterEn,

            0x10A => Self::SEnvCfg,

            0x140 => Self::SScratch,
            0x141 => Self::Sepc,
            0x142 => Self::Scause,
            0x143 => Self::Stval,
            0x144 => Self::Sip,

            0x180 => Self::Satp,

            0x5A8 => Self::SContext,
            // SUPERVISOR Mode end

            // Other
            other => Self::Other(other),
        }
    }
}

#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub enum Register {
    X0 = 0,
    X1,
    X2,
    X3,
    X4,
    X5,
    X6,
    X7,
    X8,
    X9,
    X10,
    X11,
    X12,
    X13,
    X14,
    X15,
    X16,
    X17,
    X18,
    X19,
    X20,
    X21,
    X22,
    X23,
    X24,
    X25,
    X26,
    X27,
    X28,
    X29,
    X30,
    X31,
}

impl From<u8> for Register {
    fn from(val: u8) -> Self {
        match val {
            0 | 32 => Self::X0,
            1 => Self::X1,
            2 => Self::X2,
            3 => Self::X3,
            4 => Self::X4,
            5 => Self::X5,
            6 => Self::X6,
            7 => Self::X7,
            8 => Self::X8,
            9 => Self::X9,
            10 => Self::X10,
            11 => Self::X11,
            12 => Self::X12,
            13 => Self::X13,
            14 => Self::X14,
            15 => Self::X15,
            16 => Self::X16,
            17 => Self::X17,
            18 => Self::X18,
            19 => Self::X19,
            20 => Self::X20,
            21 => Self::X21,
            22 => Self::X22,
            23 => Self::X23,
            24 => Self::X24,
            25 => Self::X25,
            26 => Self::X26,
            27 => Self::X27,
            28 => Self::X28,
            29 => Self::X29,
            30 => Self::X30,
            31 => Self::X31,
            _ => todo!(),
        }
    }
}

bitflags::bitflags! {
    pub struct FenceType : u8 {
        const NONE   = 0b0000;
        const WRITES = 0b0001;
        const READS  = 0b0010;
        const OUTPUT = 0b0100;
        const INPUT  = 0b1000;
    }
}

#[derive(Debug)]
pub enum Fence {
    General {
        succ: FenceType,
        pred: FenceType,
    },
    TSO,
    PAUSE,
    I(Register, Register, i32),
    Unknown {
        rd: Register,
        rs1: Register,
        funct3: u8,
        succ: FenceType,
        pred: FenceType,
        fm: u8,
    },
}

impl Fence {
    fn decode(inst: u32) -> Self {
        let rd = Register::from(((inst >> 7) & 0b11111) as u8);
        let rs1 = Register::from(((inst >> 15) & 0b11111) as u8);
        let funct3 = ((inst >> 12) & 0b111) as u8;
        let succ = FenceType::from_bits_truncate(((inst >> 20) & 0b1111) as u8);
        let pred = FenceType::from_bits_truncate(((inst >> 24) & 0b1111) as u8);
        let fm = ((inst >> 28) & 0b1111) as u8;
        let imm = (inst as i32) >> 20;

        match (rd, funct3, rs1, succ, pred, fm) {
            (
                Register::X0,
                0b000,
                Register::X0,
                FenceType::READS | FenceType::WRITES,
                FenceType::READS | FenceType::WRITES,
                0b1000,
            ) => Self::TSO,
            (Register::X0, 0b000, Register::X0, FenceType::NONE, FenceType::WRITES, 0b0000) => {
                Self::PAUSE
            }
            (_, 0b000, _, _, _, 0b000) => Self::General { succ, pred },
            (_, 0b001, _, _, _, _) => Self::I(rd, rs1, imm),
            (_, _, _, _, _, _) => Self::Unknown {
                rd,
                rs1,
                funct3,
                succ,
                pred,
                fm,
            },
        }
    }
}

type Imm = i32;

#[derive(Debug)]
pub enum Instruction {
    Lui(Register, u32),
    AuiPC(Register, u32),
    Jal(Register, Imm),
    Jalr(Register, Register, Imm),

    Beq(Register, Register, Imm),
    Bne(Register, Register, Imm),
    Blt(Register, Register, Imm),
    Bge(Register, Register, Imm),
    Bltu(Register, Register, Imm),
    Bgeu(Register, Register, Imm),

    Lb(Register, Register, Imm),
    Lh(Register, Register, Imm),
    Lw(Register, Register, Imm),
    Ld(Register, Register, Imm),
    Lbu(Register, Register, Imm),
    Lhu(Register, Register, Imm),
    Lwu(Register, Register, Imm),

    Sb(Register, Register, Imm),
    Sh(Register, Register, Imm),
    Sw(Register, Register, Imm),
    Sd(Register, Register, Imm),

    Addi(Register, Register, Imm),
    Addiw(Register, Register, Imm),

    Slti(Register, Register, Imm),
    Sltiu(Register, Register, Imm),
    Xori(Register, Register, Imm),
    ORI(Register, Register, Imm),
    ANDI(Register, Register, Imm),

    SLLI(Register, Register, u8),
    SLLIW(Register, Register, u8),

    SRLI(Register, Register, u8),
    SRLIW(Register, Register, u8),

    SRAI(Register, Register, u8),
    SRAIW(Register, Register, u8),

    ADD(Register, Register, Register),
    ADDW(Register, Register, Register),

    SUB(Register, Register, Register),
    SUBW(Register, Register, Register),

    SLL(Register, Register, Register),
    SLLW(Register, Register, Register),

    SLT(Register, Register, Register),
    SLTU(Register, Register, Register),
    XOR(Register, Register, Register),

    SRL(Register, Register, Register),
    SRLW(Register, Register, Register),

    SRA(Register, Register, Register),
    SRAW(Register, Register, Register),

    OR(Register, Register, Register),
    AND(Register, Register, Register),

    Mul(Register, Register, Register),
    Mulh(Register, Register, Register),
    Mulhsu(Register, Register, Register),
    Mulhu(Register, Register, Register),
    Mulw(Register, Register, Register),

    Div(Register, Register, Register),
    Divu(Register, Register, Register),
    Divw(Register, Register, Register),
    Divuw(Register, Register, Register),

    Rem(Register, Register, Register),
    Remu(Register, Register, Register),
    Remw(Register, Register, Register),
    Remuw(Register, Register, Register),

    FENCE(Fence),

    ECALL,
    EBREAK,

    CsrRw(Register, Register, Csr),
    CsrRs(Register, Register, Csr),
    CsrRc(Register, Register, Csr),
    CsrRwi(Register, u32, Csr),
    CsrRsi(Register, u32, Csr),
    CsrRci(Register, u32, Csr),

    LrW(Register, Register, Register, Ordering),
    ScW(Register, Register, Register, Ordering),
    AMOSwapW(Register, Register, Register, Ordering),
    AMOAddW(Register, Register, Register, Ordering),
    AMOXorW(Register, Register, Register, Ordering),
    AMOAndW(Register, Register, Register, Ordering),
    AMOOrW(Register, Register, Register, Ordering),
    AMOMinW(Register, Register, Register, Ordering),
    AMOMaxW(Register, Register, Register, Ordering),
    AMOMinUW(Register, Register, Register, Ordering),
    AMOMaxUW(Register, Register, Register, Ordering),

    LrD(Register, Register, Register, Ordering),
    ScD(Register, Register, Register, Ordering),
    AMOSwapD(Register, Register, Register, Ordering),
    AMOAddD(Register, Register, Register, Ordering),
    AMOXorD(Register, Register, Register, Ordering),
    AMOAndD(Register, Register, Register, Ordering),
    AMOOrD(Register, Register, Register, Ordering),
    AMOMinD(Register, Register, Register, Ordering),
    AMOMaxD(Register, Register, Register, Ordering),
    AMOMinUD(Register, Register, Register, Ordering),
    AMOMaxUD(Register, Register, Register, Ordering),
    // FlW(Register, Register, Imm),
    // FsW(Register, Register, Imm),

    // FMAddS(Register, Register, Register, Register),
    // FMSubS(Register, Register, Register, Register),
    // FNMSubS(Register, Register, Register, Register),
    // FNMAddS(Register, Register, Register, Register),
    // FAddS(Register, Register, Register),
    // FSubS(Register, Register, Register),
    // FMulS(Register, Register, Register),
    // FDivS(Register, Register, Register),
    // FSqrtS(Register, Register),
    // FSgnjS(Register, Register, Register),
    // FSgnjNS(Register, Register, Register),
    // FSgnjXS(Register, Register, Register),
    // FMinS(Register, Register, Register),
    // FMaxS(Register, Register, Register),
    // FCvtWS(Register, Register),
    // FCvtWuS(Register, Register),
    // FMvXW(Register, Register),
    // FEqS(Register, Register, Register),
    // FLtS(Register, Register, Register),
    // FLeS(Register, Register, Register),
    // FClassS(Register, Register),
    // FCvtSW(Register, Register),
    // FCvtSWU(Register, Register),
    // FMvWX(Register, Register),
    // FCvtLS(Register, Register),
    // FCvtLuS(Register, Register),
    // FCvtSL(Register, Register),
    // FCvtSLu(Register, Register),

    // FlD(Register, Register, Imm),
    // FsD(Register, Register, Imm),
    // FMAddD(Register, Register, Register, Register),
    // FMSubD(Register, Register, Register, Register),
    // FNMSubD(Register, Register, Register, Register),
    // FNMAddD(Register, Register, Register, Register),
    // FAddD(Register, Register, Register),
    // FSubD(Register, Register, Register),
    // FMulD(Register, Register, Register),
    // FDivD(Register, Register, Register),
    // FSqrtD(Register, Register),
    // FSgnjD(Register, Register, Register),
    // FSgnjND(Register, Register, Register),
    // FSgnjXD(Register, Register, Register),
    // FMinD(Register, Register, Register),
    // FMaxD(Register, Register, Register),
    // FCvtWD(Register, Register),
    // FCvtWuD(Register, Register),
    // FEqD(Register, Register, Register),
    // FLtD(Register, Register, Register),
    // FLeD(Register, Register, Register),
    // FClassD(Register, Register),
    // FCvtDW(Register, Register),
    // FCvtDWU(Register, Register),
    // FMvDX(Register, Register),
    // FCvtLD(Register, Register),
    // FCvtLuD(Register, Register),
    // FCvtDL(Register, Register),
    // FCvtDLu(Register, Register),

    // FlQ(Register, Register, Imm),
    // FsQ(Register, Register, Imm),
    // FMAddQ(Register, Register, Register, Register),
    // FMSubQ(Register, Register, Register, Register),
    // FNMSubQ(Register, Register, Register, Register),
    // FNMAddQ(Register, Register, Register, Register),
    // FAddQ(Register, Register, Register),
    // FSubQ(Register, Register, Register),
    // FMulQ(Register, Register, Register),
    // FDivQ(Register, Register, Register),
    // FSqrtQ(Register, Register),
    // FSgnjQ(Register, Register, Register),
    // FSgnjNQ(Register, Register, Register),
    // FSgnjXQ(Register, Register, Register),
    // FMinQ(Register, Register, Register),
    // FMaxQ(Register, Register, Register),
    // FCvtWQ(Register, Register),
    // FCvtWuQ(Register, Register),
    // FEqQ(Register, Register, Register),
    // FLtQ(Register, Register, Register),
    // FLeQ(Register, Register, Register),
    // FClassQ(Register, Register),
    // FCvtQW(Register, Register),
    // FCvtQWU(Register, Register),
    // FCvtLQ(Register, Register),
    // FCvtLuQ(Register, Register),
    // FCvtQL(Register, Register),
    // FCvtQLu(Register, Register),

    // FlH(Register, Register, Imm),
    // FsH(Register, Register, Imm),
}

#[test]
fn test() {
    let file = std::fs::File::open("test").unwrap(); //&[0x13, 0x01, 0x01, 0xff, 0x23, 0x24, 0xa1, 0x00];

    let dec = Decoder::new(file);

    for (i, inst) in dec.enumerate() {
        let i = i * 4;
        let i = i + 0x11d7c;
        println!("{i:#016x}: {inst:?}");
        //println!("{i:#016x}: {inst:}");
    }
}
