//! This crate provides an `EntryCustom` that directly links to libvulkan, so situations where
//! `dlopen()`-ing it is undesirable (e.g. NixOS) can be avoided.
//!
//! ## Example
//!
//! ```
//! # fn main() {
//! use ash::vk;
//! use ash_direct_entry::LibVulkan;
//!
//! let entry = LibVulkan::entry();
//!
//! let app_info = vk::ApplicationInfo {
//!     api_version: vk::make_api_version(0, 1, 0, 0),
//!     ..Default::default()
//! };
//! let create_info = vk::InstanceCreateInfo {
//!     p_application_info: &app_info,
//!     ..Default::default()
//! };
//! let instance = unsafe { entry.create_instance(&create_info, None).unwrap() };
//! # }
//! ```

use ash::EntryCustom;
use std::{ffi::c_void, os::raw::c_char, ptr};

/// A convenient alias.
pub type Entry = EntryCustom<LibVulkan>;

/// A marker type for [`EntryCustom`](https://docs.rs/ash/0.32.1/ash/struct.EntryCustom.html).
#[derive(Clone, Copy, Debug)]
pub struct LibVulkan;

impl LibVulkan {
    /// Returns an ash Entry for the libvulkan that is linked to the binary.
    ///
    /// ## Example
    ///
    /// ```
    /// # fn main() {
    /// use ash_direct_entry::LibVulkan;
    ///
    /// let entry = LibVulkan::entry();
    /// entry.fp_v1_0();
    /// entry.fp_v1_1();
    /// entry.fp_v1_2();
    /// # }
    /// ```
    pub fn entry() -> EntryCustom<LibVulkan> {
        EntryCustom::new_custom(LibVulkan, |_, name| match name.to_bytes() {
            b"vkGetInstanceProcAddr" => vkGetInstanceProcAddr as *mut c_void,
            _ => ptr::null(),
        })
        .expect("The behavior of ash changed in an unexpected way")
    }
}

#[link(name = "vulkan")]
extern "C" {
    fn vkGetInstanceProcAddr(instance: *mut c_void, name: *const c_char) -> extern "C" fn();
}
